<?php
// Enable error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Verifycertificate extends CI_Controller
{
    private $sch_setting;
    
    public function __construct()
    {
        parent::__construct();
        $this->load->helper('url');
        $this->load->helper('form');
        $this->load->library('form_validation');
        $this->load->library('session');
        $this->load->database(); // Add database library
        $this->load->model('verifycertificate_model');
        $this->load->model('setting_model');
        
        // Load school settings once for all methods
        $this->loadSchoolSettings();
    }
    
    /**
     * Load school settings and convert to object
     */
    private function loadSchoolSettings()
    {
        $settings = $this->setting_model->get();
        
        // Convert to object for view compatibility
        $this->sch_setting = new stdClass();
        if (is_array($settings) && !empty($settings)) {
            foreach ($settings[0] as $key => $value) {
                $this->sch_setting->$key = $value;
            }
        } else {
            $this->sch_setting->name = 'School Name';
            $this->sch_setting->address = 'School Address';
            $this->sch_setting->phone = '';
            $this->sch_setting->email = 'school@example.com';
            $this->sch_setting->image = 'logo.png';
        }
    }

    /**
     * Default method - shows certificate verification form
     */
    public function index()
    {
        try {
            $data = array();
            $data['title'] = 'Verify Certificate';
            $data['error'] = '';
            $data['success'] = '';
            $data['sch_setting'] = $this->sch_setting;
            
            // Try to load the view with full path for debugging
            if (file_exists(APPPATH . 'views/verify_certificate_view.php')) {
                $this->load->view('verify_certificate_view', $data);
            } else {
                echo "View file not found at: " . APPPATH . 'views/verify_certificate_view.php';
                // Try loading from themes directory as fallback
                $this->load->view('themes/default/verify_certificate', $data);
            }
        } catch (Exception $e) {
            // Display error message
            echo '<h1>An error occurred</h1>';
            echo '<p>' . $e->getMessage() . '</p>';
            echo '<p><a href="' . site_url() . '">Return to Home</a></p>';
        }
    }

    /**
     * Verify certificate by admission number
     */
    public function verify()
    {
        try {
            $this->form_validation->set_rules('admission_no', 'Admission Number', 'trim|required');
            $this->form_validation->set_rules('student_name', 'Student Name', 'trim|required');
            
            if ($this->form_validation->run() == false) {
                $data = array();
                $data['title'] = 'Verify Certificate';
                $data['error'] = validation_errors() ? validation_errors() : 'Please enter both admission number and student name.';
                $data['success'] = '';
                $data['sch_setting'] = $this->sch_setting;
                
                // Try to load the view with full path for debugging
                if (file_exists(APPPATH . 'views/verify_certificate_view.php')) {
                    $this->load->view('verify_certificate_view', $data);
                } else {
                    echo "View file not found at: " . APPPATH . 'views/verify_certificate_view.php';
                    // Try loading from themes directory as fallback
                    $this->load->view('themes/default/verify_certificate', $data);
                }
            } else {
                $admission_no = $this->input->post('admission_no');
                $student_name = $this->input->post('student_name');
                
                // Use the model to verify the student
                if (!empty($student_name)) {
                    // If student name is provided, use both admission number and name for verification
                    $student = $this->verifycertificate_model->verifyByAdmissionNoAndName($admission_no, $student_name);
                } else {
                    // Otherwise, just use admission number
                    $student = $this->verifycertificate_model->verifyByAdmissionNo($admission_no);
                }
                
                $data = array();
                $data['title'] = 'Student Verification Result';
                $data['sch_setting'] = $this->sch_setting;
                $data['admission_no'] = $admission_no;
                $data['student_name'] = $student_name;
                
                if ($student) {
                    // Add student photo URL if available
                    if (isset($student['image']) && !empty($student['image'])) {
                        $student['photo_url'] = base_url() . 'uploads/student_images/' . $student['image'];
                    } else {
                        // Default photo if not available
                        $student['photo_url'] = base_url() . 'uploads/student_images/no_image.png';
                    }
                    
                    // For debugging - log the student data
                    error_log('Student data: ' . print_r($student, true));
                    error_log('Photo URL: ' . $student['photo_url']);
                    
                    // Add a small delay to simulate verification process
                    sleep(2);
                    
                    // Determine subjects based on class
                    $student_class = isset($student['class']) ? strtolower($student['class']) : '';
                    
                    // Default subjects
                    $subjects = 'English, Hindi, Mathematics, Science';
                    
                    // Assign subjects based on class
                    if (strpos($student_class, 'nursery') !== false || strpos($student_class, 'kg') !== false || 
                        strpos($student_class, 'kindergarten') !== false) {
                        $subjects = 'English, Hindi, Mathematics, Drawing, General Knowledge';
                    } else if (strpos($student_class, '1') !== false || strpos($student_class, '2') !== false || 
                               strpos($student_class, '3') !== false || strpos($student_class, '4') !== false || 
                               strpos($student_class, '5') !== false) {
                        $subjects = 'English, Hindi, Mathematics, Science, Social Studies, General Knowledge, Computer';
                    } else if (strpos($student_class, '6') !== false || strpos($student_class, '7') !== false || 
                               strpos($student_class, '8') !== false) {
                        $subjects = 'English, Hindi, Mathematics, Science, Social Science, Sanskrit, Computer';
                    } else if (strpos($student_class, '9') !== false || strpos($student_class, '10') !== false) {
                        $subjects = 'English, Hindi, Mathematics, Science, Social Science, Information Technology';
                    } else if (strpos($student_class, '11') !== false || strpos($student_class, '12') !== false) {
                        // For 11th and 12th, check if it's a science, commerce or arts stream
                        if (strpos($student_class, 'science') !== false) {
                            $subjects = 'English, Physics, Chemistry, Mathematics/Biology, Computer Science/Physical Education';
                        } else if (strpos($student_class, 'commerce') !== false) {
                            $subjects = 'English, Accountancy, Business Studies, Economics, Mathematics/Informatics Practices';
                        } else if (strpos($student_class, 'arts') !== false || strpos($student_class, 'humanities') !== false) {
                            $subjects = 'English, History, Political Science, Geography/Psychology, Economics/Sociology';
                        } else {
                            $subjects = 'English, Physics, Chemistry, Mathematics/Biology, Computer Science/Physical Education';
                        }
                    }
                    
                    // Add subjects to student data
                    $student['subjects'] = $subjects;
                    
                    // Generate exam result information
                    $current_year = date('Y');
                    $last_year = $current_year - 1;
                    $exam_result = '';
                    
                    // Check if class contains information to determine exam type
                    if (strpos($student_class, 'nursery') !== false || strpos($student_class, 'kg') !== false || 
                        strpos($student_class, 'kindergarten') !== false || 
                        strpos($student_class, '1') !== false || strpos($student_class, '2') !== false || 
                        strpos($student_class, '3') !== false || strpos($student_class, '4') !== false || 
                        strpos($student_class, '5') !== false || strpos($student_class, '6') !== false || 
                        strpos($student_class, '7') !== false || strpos($student_class, '8') !== false) {
                        // For lower classes, show school exam
                        $exam_result = 'School Annual Examination ' . $last_year . '-' . $current_year . ', PASSED';
                    } else if (strpos($student_class, '9') !== false || strpos($student_class, '11') !== false) {
                        // For 9th and 11th, show school exam
                        $exam_result = 'School Annual Examination ' . $last_year . '-' . $current_year . ', PASSED WITH FIRST DIVISION';
                    } else if (strpos($student_class, '10') !== false) {
                        // For 10th, show board exam
                        $exam_result = 'CBSE Board Examination ' . $last_year . '-' . $current_year . ', PASSED WITH FIRST DIVISION';
                    } else if (strpos($student_class, '12') !== false) {
                        // For 12th, show board exam
                        $exam_result = 'CBSE Board Examination ' . $last_year . '-' . $current_year . ', PASSED WITH FIRST DIVISION';
                    } else {
                        // Default
                        $exam_result = 'School Annual Examination ' . $last_year . '-' . $current_year . ', PASSED';
                    }
                    
                    // Add exam result to student data
                    $student['exam_result'] = $exam_result;
                    
                    $data['success'] = 'Student verified successfully!';
                    $data['error'] = '';
                    $data['student'] = $student;
                    
                    // Set a direct photo URL that will definitely work
                    $data['student_photo'] = $student['photo_url'];
                    
                    // Also set a backup URL in case the above doesn't work
                    $data['backup_photo'] = 'https://via.placeholder.com/120x120/000000/00FF00?text=STUDENT';
                } else {
                    $data['error'] = 'Student not found. Please check the admission number and try again.';
                    $data['success'] = '';
                    $data['student'] = null;
                }
                
                // Try to load the view with full path for debugging
                if (file_exists(APPPATH . 'views/verify_certificate_view.php')) {
                    $this->load->view('verify_certificate_view', $data);
                } else {
                    echo "View file not found at: " . APPPATH . 'views/verify_certificate_view.php';
                    // Try loading from themes directory as fallback
                    $this->load->view('themes/default/verify_certificate', $data);
                }
            }
        } catch (Exception $e) {
            // Display error message
            echo '<h1>An error occurred</h1>';
            echo '<p>' . $e->getMessage() . '</p>';
            echo '<p><a href="' . site_url() . '">Return to Home</a></p>';
        }
    }
}
