<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Accounts extends Admin_Controller
{

    public function __construct()
    {
        parent::__construct();
        $this->load->model('accounts_model');
        $this->load->model('income_model');
        $this->load->model('expense_model');
        $this->load->library('form_validation');
        $this->load->helper('url');
        $this->load->helper('accounts_lang');
        
        // Load accounts language file
        load_accounts_language();
        
        $this->sch_setting_detail = $this->setting_model->getSetting();
    }

    /**
     * Main Accounts Dashboard
     */
    public function index()
    {
        if (!$this->rbac->hasPrivilege('accounts', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Accounts');
        $this->session->set_userdata('sub_menu', 'accounts/index');
        
        $data['title'] = 'Accounts Dashboard';
        
        // Get financial summary
        $data['financial_summary'] = $this->getFinancialSummary();
        
        // Get recent transactions
        $data['recent_income'] = $this->income_model->get();
        $data['recent_expenses'] = $this->expense_model->get();
        
        // Get account numbers
        $data['accounts_list'] = $this->accounts_model->getAllAccounts();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/accounts/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Account Numbers Management
     */
    public function account_numbers()
    {
        if (!$this->rbac->hasPrivilege('accounts', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Accounts');
        $this->session->set_userdata('sub_menu', 'accounts/account_numbers');
        
        $data['title'] = 'Account Numbers Management';
        
        // Handle form submission
        $this->form_validation->set_rules('account_number', 'Account Number', 'trim|required|xss_clean');
        $this->form_validation->set_rules('account_name', 'Account Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('account_type', 'Account Type', 'trim|required|xss_clean');
        
        if ($this->form_validation->run() == true) {
            $account_data = array(
                'account_number' => $this->input->post('account_number'),
                'account_name' => $this->input->post('account_name'),
                'account_type' => $this->input->post('account_type'),
                'description' => $this->input->post('description'),
                'opening_balance' => $this->input->post('opening_balance') ? convertCurrencyFormatToBaseAmount($this->input->post('opening_balance')) : 0,
                'created_date' => date('Y-m-d')
            );
            
            $insert_id = $this->accounts_model->addAccount($account_data);
            
            if ($insert_id) {
                $this->session->set_flashdata('msg', '<div class="alert alert-success">Account created successfully</div>');
            } else {
                $this->session->set_flashdata('msg', '<div class="alert alert-danger">Failed to create account</div>');
            }
            
            redirect('admin/accounts/account_numbers');
        }
        
        $data['accounts_list'] = $this->accounts_model->getAllAccounts();
        $data['account_types'] = array(
            'asset' => 'Asset',
            'liability' => 'Liability', 
            'equity' => 'Equity',
            'revenue' => 'Revenue',
            'expense' => 'Expense'
        );
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/accounts/account_numbers', $data);
        $this->load->view('layout/footer', $data);
    }


    /**
     * Balance Sheet
     */
    public function balance_sheet()
    {
        if (!$this->rbac->hasPrivilege('accounts', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Accounts');
        $this->session->set_userdata('sub_menu', 'accounts/balance_sheet');
        
        $data['title'] = 'Balance Sheet';
        
        // Get balance sheet data
        $data['assets'] = $this->accounts_model->getAccountsByType('asset');
        $data['liabilities'] = $this->accounts_model->getAccountsByType('liability');
        $data['equity'] = $this->accounts_model->getAccountsByType('equity');
        
        // Calculate totals
        $data['total_assets'] = $this->accounts_model->getTotalByType('asset');
        $data['total_liabilities'] = $this->accounts_model->getTotalByType('liability');
        $data['total_equity'] = $this->accounts_model->getTotalByType('equity');
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/accounts/balance_sheet', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Get Financial Summary
     */
    private function getFinancialSummary()
    {
        $summary = array();
        
        // Current month
        $current_month_start = date('Y-m-01');
        $current_month_end = date('Y-m-t');
        
        // Current year
        $current_year_start = date('Y-01-01');
        $current_year_end = date('Y-12-31');
        
        // Monthly totals
        $summary['monthly_income'] = $this->accounts_model->getTotalIncome($current_month_start, $current_month_end);
        $summary['monthly_expense'] = $this->accounts_model->getTotalExpense($current_month_start, $current_month_end);
        $summary['monthly_profit'] = $summary['monthly_income'] - $summary['monthly_expense'];
        
        // Yearly totals
        $summary['yearly_income'] = $this->accounts_model->getTotalIncome($current_year_start, $current_year_end);
        $summary['yearly_expense'] = $this->accounts_model->getTotalExpense($current_year_start, $current_year_end);
        $summary['yearly_profit'] = $summary['yearly_income'] - $summary['yearly_expense'];
        
        // All time totals
        $summary['total_income'] = $this->accounts_model->getTotalIncome();
        $summary['total_expense'] = $this->accounts_model->getTotalExpense();
        $summary['total_profit'] = $summary['total_income'] - $summary['total_expense'];
        
        return $summary;
    }

    /**
     * Get Income Report
     */
    private function getIncomeReport($date_from, $date_to)
    {
        return $this->accounts_model->getIncomeByCategory($date_from, $date_to);
    }

    /**
     * Get Expense Report
     */
    private function getExpenseReport($date_from, $date_to)
    {
        return $this->accounts_model->getExpenseByCategory($date_from, $date_to);
    }

    /**
     * Get Profit & Loss
     */
    private function getProfitLoss($date_from, $date_to)
    {
        $income_total = $this->accounts_model->getTotalIncome($date_from, $date_to);
        $expense_total = $this->accounts_model->getTotalExpense($date_from, $date_to);
        
        return array(
            'total_income' => $income_total,
            'total_expense' => $expense_total,
            'net_profit' => $income_total - $expense_total
        );
    }

    /**
     * Edit Account
     */
    public function edit_account($id)
    {
        if (!$this->rbac->hasPrivilege('accounts', 'can_edit')) {
            access_denied();
        }

        $account = $this->accounts_model->getAccount($id);
        
        if (!$account) {
            show_404();
        }

        $this->form_validation->set_rules('account_number', 'Account Number', 'trim|required|xss_clean');
        $this->form_validation->set_rules('account_name', 'Account Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('account_type', 'Account Type', 'trim|required|xss_clean');
        
        if ($this->form_validation->run() == true) {
            $account_data = array(
                'id' => $id,
                'account_number' => $this->input->post('account_number'),
                'account_name' => $this->input->post('account_name'),
                'account_type' => $this->input->post('account_type'),
                'description' => $this->input->post('description'),
                'opening_balance' => $this->input->post('opening_balance') ? $this->input->post('opening_balance') : 0,
                'is_active' => $this->input->post('is_active') ? $this->input->post('is_active') : 'yes'
            );
            
            $result = $this->accounts_model->updateAccount($account_data);
            
            if ($result) {
                $this->session->set_flashdata('msg', '<div class="alert alert-success">Account updated successfully</div>');
            } else {
                $this->session->set_flashdata('msg', '<div class="alert alert-danger">Failed to update account</div>');
            }
            
            redirect('admin/accounts/account_numbers');
        }

        $data['title'] = 'Edit Account';
        $data['account'] = $account;
        $data['account_types'] = array(
            'asset' => 'Asset',
            'liability' => 'Liability', 
            'equity' => 'Equity',
            'revenue' => 'Revenue',
            'expense' => 'Expense'
        );
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/accounts/edit_account', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Delete Account
     */
    public function delete_account($id)
    {
        if (!$this->rbac->hasPrivilege('accounts', 'can_delete')) {
            access_denied();
        }

        $result = $this->accounts_model->deleteAccount($id);
        
        if ($result) {
            $this->session->set_flashdata('msg', '<div class="alert alert-success">Account deleted successfully</div>');
        } else {
            $this->session->set_flashdata('msg', '<div class="alert alert-danger">Failed to delete account</div>');
        }
        
        redirect('admin/accounts/account_numbers');
    }
}
