<?php
if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Settings extends MY_Addon_QRAttendanceController
{
    function __construct()
    {
        parent::__construct();
        $this->load->model('qrattendance/qrsetting_model');
        $this->load->model('student_model');
        $this->load->model('staff_model');
        
        // Create camera settings table if it doesn't exist
        $this->createCameraSettingsTable();
    }

    public function index()
    {
        if (!$this->rbac->hasPrivilege('face_attendance', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'faceattendance');
        $this->session->set_userdata('sub_menu', 'admin/faceattendance/settings/index');

        $data = array();
        $data['page_title'] = 'Camera Settings';
        $data['version'] = $this->config->item('version');
        
        // Get current settings
        $data['settings'] = $this->getCameraSettings();
        
        // Load the view
        try {
            $this->load->view('layout/header');
            $this->load->view('admin/faceattendance/settings', $data);
            $this->load->view('layout/footer');
        } catch (Exception $e) {
            error_log('Error loading settings page: ' . $e->getMessage());
            echo '<div class="alert alert-danger">Error loading settings page. Please check error logs.</div>';
        }
    }
    
    /**
     * Get camera settings from database
     */
    public function getCameraSettings()
    {
        $this->db->where('id', 1);
        $settings = $this->db->get('camera_settings')->row();
        
        if (!$settings) {
            // Return default settings if none exist
            return (object) [
                'camera_preference' => 'user_choice',
                'face_match_threshold' => 0.6,
                'auto_capture_delay' => 3,
                'checkin_start_time' => '08:00',
                'checkin_end_time' => '10:00',
                'checkout_start_time' => '15:00',
                'checkout_end_time' => '18:00',
                'sms_notifications' => 1,
                'camera_resolution' => 'auto'
            ];
        }
        
        return $settings;
    }
    
    /**
     * Save camera settings
     */
    public function saveSettings()
    {
        // Set content type for JSON response
        header('Content-Type: application/json');
        
        // Check permission
        if (!$this->rbac->hasPrivilege('qr_code_attendance', 'can_edit')) {
            echo json_encode(['status' => false, 'msg' => 'You do not have permission to edit settings']);
            return;
        }
        
        // Get POST data
        $camera_preference = $this->input->post('camera_preference');
        $face_match_threshold = $this->input->post('face_match_threshold');
        $auto_capture_delay = $this->input->post('auto_capture_delay');
        $checkin_start_time = $this->input->post('checkin_start_time');
        $checkin_end_time = $this->input->post('checkin_end_time');
        $checkout_start_time = $this->input->post('checkout_start_time');
        $checkout_end_time = $this->input->post('checkout_end_time');
        $sms_notifications = $this->input->post('sms_notifications') ? 1 : 0;
        $camera_resolution = $this->input->post('camera_resolution');
        
        // Validate data
        if (empty($camera_preference) || empty($face_match_threshold) || empty($auto_capture_delay)) {
            echo json_encode(['status' => false, 'msg' => 'Please fill all required fields']);
            return;
        }
        
        // Validate face match threshold
        if ($face_match_threshold < 0.3 || $face_match_threshold > 0.9) {
            echo json_encode(['status' => false, 'msg' => 'Face match threshold must be between 0.3 and 0.9']);
            return;
        }
        
        // Validate auto capture delay
        if ($auto_capture_delay < 1 || $auto_capture_delay > 10) {
            echo json_encode(['status' => false, 'msg' => 'Auto capture delay must be between 1 and 10 seconds']);
            return;
        }
        
        // Validate time format
        if (!$this->validateTimeFormat($checkin_start_time) || !$this->validateTimeFormat($checkin_end_time) ||
            !$this->validateTimeFormat($checkout_start_time) || !$this->validateTimeFormat($checkout_end_time)) {
            echo json_encode(['status' => false, 'msg' => 'Invalid time format. Please use HH:MM format']);
            return;
        }
        
        // Prepare data for database
        $settings_data = [
            'camera_preference' => $camera_preference,
            'face_match_threshold' => floatval($face_match_threshold),
            'auto_capture_delay' => intval($auto_capture_delay),
            'checkin_start_time' => $checkin_start_time,
            'checkin_end_time' => $checkin_end_time,
            'checkout_start_time' => $checkout_start_time,
            'checkout_end_time' => $checkout_end_time,
            'sms_notifications' => $sms_notifications,
            'camera_resolution' => $camera_resolution,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        try {
            // Check if settings exist
            $this->db->where('id', 1);
            $existing = $this->db->get('camera_settings')->row();
            
            if ($existing) {
                // Update existing settings
                $this->db->where('id', 1);
                $result = $this->db->update('camera_settings', $settings_data);
            } else {
                // Insert new settings
                $settings_data['id'] = 1;
                $settings_data['created_at'] = date('Y-m-d H:i:s');
                $result = $this->db->insert('camera_settings', $settings_data);
            }
            
            if ($result) {
                echo json_encode(['status' => true, 'msg' => 'Settings saved successfully']);
            } else {
                echo json_encode(['status' => false, 'msg' => 'Failed to save settings']);
            }
        } catch (Exception $e) {
            error_log('Error saving camera settings: ' . $e->getMessage());
            echo json_encode(['status' => false, 'msg' => 'Database error: ' . $e->getMessage()]);
        }
    }
    
    /**
     * Get settings as JSON for AJAX requests
     */
    public function getSettingsJson()
    {
        header('Content-Type: application/json');
        
        $settings = $this->getCameraSettings();
        echo json_encode(['status' => true, 'settings' => $settings]);
    }
    
    /**
     * Reset settings to default
     */
    public function resetSettings()
    {
        header('Content-Type: application/json');
        
        // Check permission
        if (!$this->rbac->hasPrivilege('qr_code_attendance', 'can_edit')) {
            echo json_encode(['status' => false, 'msg' => 'You do not have permission to reset settings']);
            return;
        }
        
        // Default settings
        $default_settings = [
            'id' => 1,
            'camera_preference' => 'user_choice',
            'face_match_threshold' => 0.6,
            'auto_capture_delay' => 3,
            'checkin_start_time' => '08:00',
            'checkin_end_time' => '10:00',
            'checkout_start_time' => '15:00',
            'checkout_end_time' => '18:00',
            'sms_notifications' => 1,
            'camera_resolution' => 'auto',
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        try {
            // Delete existing settings
            $this->db->where('id', 1);
            $this->db->delete('camera_settings');
            
            // Insert default settings
            $result = $this->db->insert('camera_settings', $default_settings);
            
            if ($result) {
                echo json_encode(['status' => true, 'msg' => 'Settings reset to default successfully']);
            } else {
                echo json_encode(['status' => false, 'msg' => 'Failed to reset settings']);
            }
        } catch (Exception $e) {
            error_log('Error resetting camera settings: ' . $e->getMessage());
            echo json_encode(['status' => false, 'msg' => 'Database error: ' . $e->getMessage()]);
        }
    }
    
    /**
     * Validate time format (HH:MM or HH:MM:SS)
     */
    private function validateTimeFormat($time)
    {
        // Check if time is empty or null
        if (empty($time)) {
            return false;
        }
        
        // Accept both HH:MM and HH:MM:SS formats
        if (!preg_match('/^([0-2][0-9]):[0-5][0-9](:[0-5][0-9])?$/', $time)) {
            return false;
        }
        
        // Extract hours and minutes (ignore seconds if present)
        $timeParts = explode(':', $time);
        $hours = intval($timeParts[0]);
        $minutes = intval($timeParts[1]);
        
        // Validate range
        return ($hours >= 0 && $hours <= 23) && ($minutes >= 0 && $minutes <= 59);
    }
    
    /**
     * Create camera_settings table if it doesn't exist
     */
    private function createCameraSettingsTable()
    {
        if (!$this->db->table_exists('camera_settings')) {
            try {
                $this->db->query("
                    CREATE TABLE `camera_settings` (
                      `id` int(11) NOT NULL AUTO_INCREMENT,
                      `camera_preference` varchar(50) NOT NULL DEFAULT 'user_choice',
                      `face_match_threshold` decimal(3,2) NOT NULL DEFAULT '0.60',
                      `auto_capture_delay` int(11) NOT NULL DEFAULT '3',
                      `checkin_start_time` time NOT NULL DEFAULT '08:00:00',
                      `checkin_end_time` time NOT NULL DEFAULT '10:00:00',
                      `checkout_start_time` time NOT NULL DEFAULT '15:00:00',
                      `checkout_end_time` time NOT NULL DEFAULT '18:00:00',
                      `sms_notifications` tinyint(1) NOT NULL DEFAULT '1',
                      `camera_resolution` varchar(20) NOT NULL DEFAULT 'auto',
                      `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                      `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                      PRIMARY KEY (`id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
                ");
                
                // Insert default settings
                $this->db->query("
                    INSERT INTO `camera_settings` (`id`, `camera_preference`, `face_match_threshold`, `auto_capture_delay`, 
                    `checkin_start_time`, `checkin_end_time`, `checkout_start_time`, `checkout_end_time`, 
                    `sms_notifications`, `camera_resolution`) VALUES 
                    (1, 'user_choice', 0.60, 3, '08:00:00', '10:00:00', '15:00:00', '18:00:00', 1, 'auto');
                ");
                
                error_log('Camera settings table created successfully');
            } catch (Exception $e) {
                error_log('Error creating camera_settings table: ' . $e->getMessage());
            }
        }
    }
    
    /**
     * Check if current time is within allowed check-in window
     */
    public function isCheckinAllowed()
    {
        header('Content-Type: application/json');
        
        $settings = $this->getCameraSettings();
        $current_time = date('H:i');
        
        $checkin_start = date('H:i', strtotime($settings->checkin_start_time));
        $checkin_end = date('H:i', strtotime($settings->checkin_end_time));
        
        $allowed = ($current_time >= $checkin_start && $current_time <= $checkin_end);
        
        echo json_encode([
            'status' => true,
            'allowed' => $allowed,
            'current_time' => $current_time,
            'checkin_window' => $checkin_start . ' - ' . $checkin_end
        ]);
    }
    
    /**
     * Check if current time is within allowed check-out window
     */
    public function isCheckoutAllowed()
    {
        header('Content-Type: application/json');
        
        $settings = $this->getCameraSettings();
        $current_time = date('H:i');
        
        $checkout_start = date('H:i', strtotime($settings->checkout_start_time));
        $checkout_end = date('H:i', strtotime($settings->checkout_end_time));
        
        $allowed = ($current_time >= $checkout_start && $current_time <= $checkout_end);
        
        echo json_encode([
            'status' => true,
            'allowed' => $allowed,
            'current_time' => $current_time,
            'checkout_window' => $checkout_start . ' - ' . $checkout_end
        ]);
    }
}
