<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Inventory_model extends CI_Model
{
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    // Dashboard Statistics
    public function getTotalItems()
    {
        return $this->db->where('is_active', 1)->count_all_results('inventory_items');
    }

    public function getTotalCategories()
    {
        return $this->db->where('is_active', 1)->count_all_results('inventory_categories');
    }

    public function getTotalSuppliers()
    {
        return $this->db->where('is_active', 1)->count_all_results('inventory_suppliers');
    }

    public function getLowStockItems()
    {
        $this->db->select('*');
        $this->db->from('inventory_items');
        $this->db->where('is_active', 1);
        $this->db->where('current_stock <= minimum_stock');
        $this->db->where('minimum_stock > 0');
        return $this->db->get()->result();
    }

    public function getTotalStockValue()
    {
        $this->db->select('SUM(current_stock * purchase_price) as total_value');
        $this->db->from('inventory_items');
        $this->db->where('is_active', 1);
        $result = $this->db->get()->row();
        return $result ? $result->total_value : 0;
    }

    public function getRecentMovements($limit = 10)
    {
        $this->db->select('sm.*, i.name as item_name, i.item_code, s.name as staff_name');
        $this->db->from('inventory_stock_movements sm');
        $this->db->join('inventory_items i', 'i.id = sm.item_id');
        $this->db->join('staff s', 's.id = sm.staff_id', 'left');
        $this->db->order_by('sm.created_at', 'DESC');
        $this->db->limit($limit);
        return $this->db->get()->result();
    }

    public function getPendingRequisitions()
    {
        $this->db->select('COUNT(*) as count');
        $this->db->from('inventory_requisitions');
        $this->db->where('status', 'PENDING');
        $result = $this->db->get()->row();
        return $result ? $result->count : 0;
    }

    public function getActiveAlerts()
    {
        $this->db->select('COUNT(*) as count');
        $this->db->from('inventory_stock_alerts');
        $this->db->where('is_acknowledged', 0);
        $result = $this->db->get()->row();
        return $result ? $result->count : 0;
    }

    // Items Management
    public function getAllItems()
    {
        $this->db->select('i.*, c.name as category_name, s.name as supplier_name');
        $this->db->from('inventory_items i');
        $this->db->join('inventory_categories c', 'c.id = i.category_id', 'left');
        $this->db->join('inventory_suppliers s', 's.id = i.supplier_id', 'left');
        $this->db->order_by('i.name', 'ASC');
        return $this->db->get()->result();
    }

    public function getItem($id)
    {
        $this->db->select('i.*, c.name as category_name, s.name as supplier_name');
        $this->db->from('inventory_items i');
        $this->db->join('inventory_categories c', 'c.id = i.category_id', 'left');
        $this->db->join('inventory_suppliers s', 's.id = i.supplier_id', 'left');
        $this->db->where('i.id', $id);
        return $this->db->get()->row();
    }

    public function addItem($data)
    {
        if ($this->db->insert('inventory_items', $data)) {
            return $this->db->insert_id();
        }
        return false;
    }

    public function updateItem($id, $data)
    {
        $this->db->where('id', $id);
        return $this->db->update('inventory_items', $data);
    }

    public function deleteItem($id)
    {
        // Check if item has any stock movements
        $this->db->where('item_id', $id);
        $movements = $this->db->count_all_results('inventory_stock_movements');
        
        if ($movements > 0) {
            // Soft delete - mark as inactive
            return $this->updateItem($id, array('is_active' => 0));
        } else {
            // Hard delete if no movements
            $this->db->where('id', $id);
            return $this->db->delete('inventory_items');
        }
    }

    public function checkItemCodeExists($item_code, $exclude_id = null)
    {
        $this->db->where('item_code', $item_code);
        if ($exclude_id) {
            $this->db->where('id !=', $exclude_id);
        }
        return $this->db->count_all_results('inventory_items') > 0;
    }

    public function getCurrentStock($item_id)
    {
        $this->db->select('current_stock');
        $this->db->where('id', $item_id);
        $result = $this->db->get('inventory_items')->row();
        return $result ? $result->current_stock : 0;
    }

    // Categories Management
    public function getAllCategories()
    {
        $this->db->select('c.*, parent.name as parent_name');
        $this->db->from('inventory_categories c');
        $this->db->join('inventory_categories parent', 'parent.id = c.parent_id', 'left');
        $this->db->order_by('c.name', 'ASC');
        return $this->db->get()->result();
    }

    public function addCategory($data)
    {
        return $this->db->insert('inventory_categories', $data);
    }

    public function updateCategory($id, $data)
    {
        $this->db->where('id', $id);
        return $this->db->update('inventory_categories', $data);
    }

    public function deleteCategory($id)
    {
        // Check if category has items
        $this->db->where('category_id', $id);
        $items = $this->db->count_all_results('inventory_items');
        
        if ($items > 0) {
            return false; // Cannot delete category with items
        }
        
        $this->db->where('id', $id);
        return $this->db->delete('inventory_categories');
    }

    // Suppliers Management
    public function getAllSuppliers()
    {
        $this->db->order_by('name', 'ASC');
        return $this->db->get('inventory_suppliers')->result();
    }

    public function getSupplier($id)
    {
        $this->db->where('id', $id);
        return $this->db->get('inventory_suppliers')->row();
    }

    public function addSupplier($data)
    {
        return $this->db->insert('inventory_suppliers', $data);
    }

    public function updateSupplier($id, $data)
    {
        $this->db->where('id', $id);
        return $this->db->update('inventory_suppliers', $data);
    }

    public function deleteSupplier($id)
    {
        // Check if supplier has items
        $this->db->where('supplier_id', $id);
        $items = $this->db->count_all_results('inventory_items');
        
        if ($items > 0) {
            // Soft delete - mark as inactive
            return $this->updateSupplier($id, array('is_active' => 0));
        } else {
            $this->db->where('id', $id);
            return $this->db->delete('inventory_suppliers');
        }
    }

    // Stock Movements
    public function getAllMovements()
    {
        $this->db->select('sm.*, i.name as item_name, i.item_code, s.name as staff_name');
        $this->db->from('inventory_stock_movements sm');
        $this->db->join('inventory_items i', 'i.id = sm.item_id');
        $this->db->join('staff s', 's.id = sm.staff_id', 'left');
        $this->db->order_by('sm.movement_date', 'DESC');
        $this->db->order_by('sm.created_at', 'DESC');
        return $this->db->get()->result();
    }

    public function addStockMovement($data)
    {
        return $this->db->insert('inventory_stock_movements', $data);
    }

    public function updateItemStock($item_id)
    {
        // Calculate current stock based on movements
        $this->db->select('
            SUM(CASE WHEN movement_type IN ("IN", "ADJUSTMENT") THEN quantity ELSE 0 END) -
            SUM(CASE WHEN movement_type IN ("OUT", "TRANSFER") THEN quantity ELSE 0 END) as calculated_stock
        ');
        $this->db->where('item_id', $item_id);
        $result = $this->db->get('inventory_stock_movements')->row();
        
        $current_stock = $result ? $result->calculated_stock : 0;
        
        // Update item stock
        $this->db->where('id', $item_id);
        return $this->db->update('inventory_items', array('current_stock' => $current_stock));
    }

    // Stock Alerts
    public function checkStockAlerts($item_id)
    {
        $item = $this->getItem($item_id);
        if (!$item) return;

        $alerts_to_create = array();
        
        // Check for low stock
        if ($item->minimum_stock > 0 && $item->current_stock <= $item->minimum_stock) {
            if ($item->current_stock == 0) {
                $alerts_to_create[] = array(
                    'item_id' => $item_id,
                    'alert_type' => 'OUT_OF_STOCK',
                    'current_stock' => $item->current_stock,
                    'threshold_value' => 0,
                    'alert_date' => date('Y-m-d')
                );
            } else {
                $alerts_to_create[] = array(
                    'item_id' => $item_id,
                    'alert_type' => 'LOW_STOCK',
                    'current_stock' => $item->current_stock,
                    'threshold_value' => $item->minimum_stock,
                    'alert_date' => date('Y-m-d')
                );
            }
        }

        // Check for overstock
        if ($item->maximum_stock > 0 && $item->current_stock > $item->maximum_stock) {
            $alerts_to_create[] = array(
                'item_id' => $item_id,
                'alert_type' => 'OVERSTOCK',
                'current_stock' => $item->current_stock,
                'threshold_value' => $item->maximum_stock,
                'alert_date' => date('Y-m-d')
            );
        }

        // Insert alerts (avoid duplicates)
        foreach ($alerts_to_create as $alert) {
            $this->db->where('item_id', $alert['item_id']);
            $this->db->where('alert_type', $alert['alert_type']);
            $this->db->where('alert_date', $alert['alert_date']);
            $this->db->where('is_acknowledged', 0);
            
            if ($this->db->count_all_results('inventory_stock_alerts') == 0) {
                $this->db->insert('inventory_stock_alerts', $alert);
            }
        }
    }

    public function getAllAlerts()
    {
        $this->db->select('a.*, i.name as item_name, i.item_code');
        $this->db->from('inventory_stock_alerts a');
        $this->db->join('inventory_items i', 'i.id = a.item_id');
        $this->db->order_by('a.alert_date', 'DESC');
        return $this->db->get()->result();
    }

    public function acknowledgeAlert($alert_id, $staff_id)
    {
        $data = array(
            'is_acknowledged' => 1,
            'acknowledged_by' => $staff_id,
            'acknowledged_at' => date('Y-m-d H:i:s')
        );
        
        $this->db->where('id', $alert_id);
        return $this->db->update('inventory_stock_alerts', $data);
    }

    // Requisitions
    public function getAllRequisitions()
    {
        $this->db->select('r.*, s1.name as requested_by_name, s2.name as approved_by_name, s3.name as issued_by_name');
        $this->db->from('inventory_requisitions r');
        $this->db->join('staff s1', 's1.id = r.requested_by', 'left');
        $this->db->join('staff s2', 's2.id = r.approved_by', 'left');
        $this->db->join('staff s3', 's3.id = r.issued_by', 'left');
        $this->db->order_by('r.request_date', 'DESC');
        return $this->db->get()->result();
    }

    public function addRequisition($data)
    {
        if ($this->db->insert('inventory_requisitions', $data)) {
            return $this->db->insert_id();
        }
        return false;
    }

    public function addRequisitionItem($data)
    {
        return $this->db->insert('inventory_requisition_items', $data);
    }

    public function updateRequisition($id, $data)
    {
        $this->db->where('id', $id);
        $result = $this->db->update('inventory_requisitions', $data);
        
        // Add debugging
        if (!$result) {
            error_log("Database update failed. Last query: " . $this->db->last_query());
            error_log("Database error: " . $this->db->error()['message']);
        }
        
        return $result;
    }

    public function getRequisitionDetails($id)
    {
        $this->db->select('r.*, s1.name as requested_by_name, s2.name as approved_by_name, s3.name as issued_by_name');
        $this->db->from('inventory_requisitions r');
        $this->db->join('staff s1', 's1.id = r.requested_by', 'left');
        $this->db->join('staff s2', 's2.id = r.approved_by', 'left');
        $this->db->join('staff s3', 's3.id = r.issued_by', 'left');
        $this->db->where('r.id', $id);
        return $this->db->get()->row();
    }

    public function getRequisitionItems($requisition_id)
    {
        $this->db->select('ri.*, i.name as item_name, i.item_code, i.current_stock');
        $this->db->from('inventory_requisition_items ri');
        $this->db->join('inventory_items i', 'i.id = ri.item_id');
        $this->db->where('ri.requisition_id', $requisition_id);
        $this->db->order_by('i.name', 'ASC');
        return $this->db->get()->result();
    }

    // Purchase Orders
    public function getAllPurchaseOrders()
    {
        $this->db->select('po.*, s.name as supplier_name, st1.name as created_by_name, st2.name as approved_by_name');
        $this->db->from('inventory_purchase_orders po');
        $this->db->join('inventory_suppliers s', 's.id = po.supplier_id');
        $this->db->join('staff st1', 'st1.id = po.created_by', 'left');
        $this->db->join('staff st2', 'st2.id = po.approved_by', 'left');
        $this->db->order_by('po.order_date', 'DESC');
        return $this->db->get()->result();
    }

    public function addPurchaseOrder($data)
    {
        return $this->db->insert('inventory_purchase_orders', $data);
    }

    public function updatePurchaseOrder($id, $data)
    {
        $this->db->where('id', $id);
        return $this->db->update('inventory_purchase_orders', $data);
    }

    // Reports
    public function getStockReport($date_from = null, $date_to = null)
    {
        $this->db->select('i.*, c.name as category_name, s.name as supplier_name');
        $this->db->from('inventory_items i');
        $this->db->join('inventory_categories c', 'c.id = i.category_id', 'left');
        $this->db->join('inventory_suppliers s', 's.id = i.supplier_id', 'left');
        $this->db->where('i.is_active', 1);
        $this->db->order_by('i.name', 'ASC');
        return $this->db->get()->result();
    }

    public function getMovementReport($date_from, $date_to, $item_id = null, $movement_type = null)
    {
        $this->db->select('sm.*, i.name as item_name, i.item_code, s.name as staff_name');
        $this->db->from('inventory_stock_movements sm');
        $this->db->join('inventory_items i', 'i.id = sm.item_id');
        $this->db->join('staff s', 's.id = sm.staff_id', 'left');
        
        if ($date_from) {
            $this->db->where('sm.movement_date >=', $date_from);
        }
        if ($date_to) {
            $this->db->where('sm.movement_date <=', $date_to);
        }
        if ($item_id) {
            $this->db->where('sm.item_id', $item_id);
        }
        if ($movement_type) {
            $this->db->where('sm.movement_type', $movement_type);
        }
        
        $this->db->order_by('sm.movement_date', 'DESC');
        $this->db->order_by('sm.created_at', 'DESC');
        return $this->db->get()->result();
    }

    public function getValuationReport()
    {
        $this->db->select('
            i.id, i.item_code, i.name, i.current_stock, i.purchase_price,
            (i.current_stock * i.purchase_price) as total_value,
            c.name as category_name
        ');
        $this->db->from('inventory_items i');
        $this->db->join('inventory_categories c', 'c.id = i.category_id', 'left');
        $this->db->where('i.is_active', 1);
        $this->db->where('i.current_stock >', 0);
        $this->db->order_by('total_value', 'DESC');
        return $this->db->get()->result();
    }

    public function getLowStockReport()
    {
        try {
            $this->db->select('
                i.id, i.item_code, i.name, i.current_stock, i.reorder_level,
                i.purchase_price, (i.current_stock * i.purchase_price) as total_value,
                c.name as category_name, s.name as supplier_name
            ');
            $this->db->from('inventory_items i');
            $this->db->join('inventory_categories c', 'c.id = i.category_id', 'left');
            $this->db->join('inventory_suppliers s', 's.id = i.supplier_id', 'left');
            $this->db->where('i.is_active', 1);
            $this->db->where('i.current_stock <=', 'i.reorder_level', FALSE);
            $this->db->order_by('i.current_stock', 'ASC');
            
            $query = $this->db->get();
            
            if ($this->db->error()['code'] != 0) {
                error_log("Database error in getLowStockReport: " . $this->db->error()['message']);
                return array();
            }
            
            return $query->result();
            
        } catch (Exception $e) {
            error_log("Exception in getLowStockReport: " . $e->getMessage());
            return array();
        }
    }

    public function getQuickStats()
    {
        try {
            // Total items count
            $this->db->select('COUNT(*) as total_items');
            $this->db->from('inventory_items');
            $this->db->where('is_active', 1);
            $total_items = $this->db->get()->row()->total_items;

            // Low stock items count
            $this->db->select('COUNT(*) as low_stock_count');
            $this->db->from('inventory_items');
            $this->db->where('current_stock <=', 'reorder_level', FALSE);
            $this->db->where('is_active', 1);
            $low_stock_count = $this->db->get()->row()->low_stock_count;

            // Total stock value
            $this->db->select('SUM(current_stock * purchase_price) as total_value');
            $this->db->from('inventory_items');
            $this->db->where('is_active', 1);
            $total_value = $this->db->get()->row()->total_value ?? 0;

            return array(
                'total_items' => (int)$total_items,
                'low_stock_count' => (int)$low_stock_count,
                'total_stock_value' => (float)$total_value
            );
        } catch (Exception $e) {
            error_log("Get quick stats error: " . $e->getMessage());
            return array(
                'total_items' => 0,
                'low_stock_count' => 0,
                'total_stock_value' => 0
            );
        }
    }

    // Utility methods
    public function generateItemCode($prefix = 'ITM')
    {
        $this->db->select('item_code');
        $this->db->like('item_code', $prefix, 'after');
        $this->db->order_by('id', 'DESC');
        $this->db->limit(1);
        $result = $this->db->get('inventory_items')->row();
        
        if ($result) {
            $last_number = (int) substr($result->item_code, strlen($prefix));
            $new_number = $last_number + 1;
        } else {
            $new_number = 1;
        }
        
        return $prefix . str_pad($new_number, 6, '0', STR_PAD_LEFT);
    }

    public function generateRequisitionNumber($prefix = 'REQ')
    {
        $this->db->select('requisition_number');
        $this->db->like('requisition_number', $prefix, 'after');
        $this->db->order_by('id', 'DESC');
        $this->db->limit(1);
        $result = $this->db->get('inventory_requisitions')->row();
        
        if ($result) {
            $last_number = (int) substr($result->requisition_number, strlen($prefix));
            $new_number = $last_number + 1;
        } else {
            $new_number = 1;
        }
        
        return $prefix . date('Y') . str_pad($new_number, 4, '0', STR_PAD_LEFT);
    }

    public function generatePONumber($prefix = 'PO')
    {
        $this->db->select('po_number');
        $this->db->like('po_number', $prefix, 'after');
        $this->db->order_by('id', 'DESC');
        $this->db->limit(1);
        $result = $this->db->get('inventory_purchase_orders')->row();
        
        if ($result) {
            $last_number = (int) substr($result->po_number, strlen($prefix));
            $new_number = $last_number + 1;
        } else {
            $new_number = 1;
        }
        
        return $prefix . date('Y') . str_pad($new_number, 4, '0', STR_PAD_LEFT);
    }

    // Student Sales Methods
    public function getAllStudentSales()
    {
        $this->db->select('
            ss.*, 
            CONCAT(st.firstname, " ", st.lastname) as student_name,
            st.admission_no,
            c.class,
            sec.section,
            COUNT(ssi.id) as total_items,
            staff.name as staff_name
        ');
        $this->db->from('student_sales ss');
        $this->db->join('students st', 'st.id = ss.student_id');
        $this->db->join('student_session sts', 'sts.student_id = st.id');
        $this->db->join('classes c', 'c.id = sts.class_id');
        $this->db->join('sections sec', 'sec.id = sts.section_id');
        $this->db->join('student_sale_items ssi', 'ssi.student_sale_id = ss.id', 'left');
        $this->db->join('staff', 'staff.id = ss.staff_id', 'left');
        $this->db->group_by('ss.id');
        $this->db->order_by('ss.sale_date', 'DESC');
        $this->db->order_by('ss.created_at', 'DESC');
        return $this->db->get()->result();
    }

    public function getActiveItems()
    {
        $this->db->select('id, item_code, name, current_stock, selling_price');
        $this->db->where('is_active', 1);
        $this->db->where('current_stock >', 0);
        $this->db->order_by('name', 'ASC');
        return $this->db->get('inventory_items')->result();
    }

    public function getNextStudentSaleNumber()
    {
        $current_year = date('Y');
        
        // Get or create counter for current year
        $this->db->where('year', $current_year);
        $counter = $this->db->get('student_sale_counter')->row();
        
        if (!$counter) {
            // Create new counter for current year
            $this->db->insert('student_sale_counter', array('year' => $current_year, 'counter' => 1));
            return 1;
        } else {
            // Increment counter
            $new_counter = $counter->counter + 1;
            $this->db->where('year', $current_year);
            $this->db->update('student_sale_counter', array('counter' => $new_counter));
            return $new_counter;
        }
    }

    public function createStudentSale($data)
    {
        if ($this->db->insert('student_sales', $data)) {
            return $this->db->insert_id();
        }
        return false;
    }

    public function addStudentSaleItem($data)
    {
        return $this->db->insert('student_sale_items', $data);
    }

    public function getStudentSale($id)
    {
        $this->db->where('id', $id);
        return $this->db->get('student_sales')->row();
    }


    public function getStudentSaleItems($sale_id)
    {
        $this->db->select('ssi.*, i.item_code, i.name as item_name, i.unit_of_measure');
        $this->db->from('student_sale_items ssi');
        $this->db->join('inventory_items i', 'i.id = ssi.item_id');
        $this->db->where('ssi.student_sale_id', $sale_id);
        $this->db->order_by('i.name', 'ASC');
        return $this->db->get()->result();
    }

    public function updateStudentSale($id, $data)
    {
        $this->db->where('id', $id);
        return $this->db->update('student_sales', $data);
    }

    public function addStudentSalePayment($data)
    {
        return $this->db->insert('student_sale_payments', $data);
    }

    public function getStudentSalePayments($sale_id)
    {
        $this->db->select('ssp.*, staff.name as staff_name');
        $this->db->from('student_sale_payments ssp');
        $this->db->join('staff', 'staff.id = ssp.staff_id', 'left');
        $this->db->where('ssp.student_sale_id', $sale_id);
        $this->db->order_by('ssp.payment_date', 'DESC');
        return $this->db->get()->result();
    }

    // Student Sales Reports
    public function getStudentSalesReport($date_from = null, $date_to = null, $class_id = null, $section_id = null)
    {
        $this->db->select('
            ss.*, 
            CONCAT(st.firstname, " ", st.lastname) as student_name,
            st.admission_no,
            c.class,
            sec.section,
            COUNT(ssi.id) as total_items,
            staff.name as staff_name
        ');
        $this->db->from('student_sales ss');
        $this->db->join('students st', 'st.id = ss.student_id');
        $this->db->join('student_session sts', 'sts.student_id = st.id');
        $this->db->join('classes c', 'c.id = sts.class_id');
        $this->db->join('sections sec', 'sec.id = sts.section_id');
        $this->db->join('student_sale_items ssi', 'ssi.student_sale_id = ss.id', 'left');
        $this->db->join('staff', 'staff.id = ss.staff_id', 'left');
        
        if ($date_from) {
            $this->db->where('ss.sale_date >=', $date_from);
        }
        if ($date_to) {
            $this->db->where('ss.sale_date <=', $date_to);
        }
        if ($class_id) {
            $this->db->where('sts.class_id', $class_id);
        }
        if ($section_id) {
            $this->db->where('sts.section_id', $section_id);
        }
        
        $this->db->group_by('ss.id');
        $this->db->order_by('ss.sale_date', 'DESC');
        return $this->db->get()->result();
    }

    // Alert Management
    public function getAlertById($id)
    {
        $this->db->select('sa.*, i.name as item_name, i.item_code');
        $this->db->from('inventory_stock_alerts sa');
        $this->db->join('inventory_items i', 'i.id = sa.item_id');
        $this->db->where('sa.id', $id);
        return $this->db->get()->row();
    }

    public function updateAlert($id, $data)
    {
        $this->db->where('id', $id);
        $result = $this->db->update('inventory_stock_alerts', $data);
        
        // Add debugging
        if (!$result) {
            error_log("Alert update failed. Last query: " . $this->db->last_query());
            error_log("Database error: " . $this->db->error()['message']);
        }
        
        return $result;
    }

    public function acknowledgeAllAlerts($data)
    {
        $this->db->where('is_acknowledged', 0);
        $result = $this->db->update('inventory_stock_alerts', $data);
        
        // Add debugging
        if (!$result) {
            error_log("Acknowledge all alerts failed. Last query: " . $this->db->last_query());
            error_log("Database error: " . $this->db->error()['message']);
        }
        
        return $result;
    }

    public function getItemById($id)
    {
        try {
            error_log("Getting item by ID: $id");
            
            $this->db->select('i.*, c.name as category_name');
            $this->db->from('inventory_items i');
            $this->db->join('inventory_categories c', 'c.id = i.category_id', 'left');
            $this->db->where('i.id', $id);
            
            $query = $this->db->get();
            error_log("Query executed: " . $this->db->last_query());
            
            if ($this->db->error()['code'] != 0) {
                error_log("Database error in getItemById: " . $this->db->error()['message']);
                return null;
            }
            
            $result = $query->row();
            error_log("Item found: " . ($result ? 'yes' : 'no'));
            
            return $result;
            
        } catch (Exception $e) {
            error_log("Exception in getItemById: " . $e->getMessage());
            return null;
        }
    }

    public function getItemMovements($item_id, $limit = null)
    {
        try {
            error_log("Getting item movements for item ID: $item_id");
            
            $this->db->select('sm.*, s.name as staff_name');
            $this->db->from('inventory_stock_movements sm');
            $this->db->join('staff s', 's.id = sm.created_by', 'left');
            $this->db->where('sm.item_id', $item_id);
            $this->db->order_by('sm.movement_date', 'DESC');
            $this->db->order_by('sm.created_at', 'DESC');
            
            if ($limit) {
                $this->db->limit($limit);
            }
            
            $query = $this->db->get();
            error_log("Movements query executed: " . $this->db->last_query());
            
            if ($this->db->error()['code'] != 0) {
                error_log("Database error in getItemMovements: " . $this->db->error()['message']);
                return array();
            }
            
            $result = $query->result();
            error_log("Movements found: " . count($result));
            
            return $result;
            
        } catch (Exception $e) {
            error_log("Exception in getItemMovements: " . $e->getMessage());
            return array();
        }
    }

    public function getStudentSalesStats($date_from = null, $date_to = null)
    {
        $this->db->select('
            COUNT(ss.id) as total_sales,
            SUM(ss.total_amount) as total_revenue,
            SUM(ss.paid_amount) as total_collected,
            SUM(ss.total_amount - ss.paid_amount) as total_pending,
            AVG(ss.total_amount) as average_sale_amount
        ');
        $this->db->from('student_sales ss');
        
        if ($date_from) {
            $this->db->where('ss.sale_date >=', $date_from);
        }
        if ($date_to) {
            $this->db->where('ss.sale_date <=', $date_to);
        }
        
        return $this->db->get()->row();
    }

    // Student Sales History Methods
    public function getStudentSalesHistory($filters = array(), $limit = 0, $offset = 0)
    {
        try {
            $this->db->select('
                ss.id as sale_id,
                ss.sale_date,
                ss.total_amount,
                ss.paid_amount,
                ss.payment_status,
                ss.payment_method,
                CONCAT(s.firstname, " ", s.lastname) as student_name,
                s.admission_no,
                c.class as class,
                sec.section as section,
                COUNT(ssi.id) as total_items
            ');
            $this->db->from('student_sales ss');
            $this->db->join('students s', 's.id = ss.student_id', 'left');
            $this->db->join('classes c', 'c.id = s.class_id', 'left');
            $this->db->join('sections sec', 'sec.id = s.section_id', 'left');
            $this->db->join('student_sale_items ssi', 'ssi.sale_id = ss.id', 'left');
            
            // Apply filters
            if (!empty($filters['date_from'])) {
                $this->db->where('ss.sale_date >=', $filters['date_from']);
            }
            if (!empty($filters['date_to'])) {
                $this->db->where('ss.sale_date <=', $filters['date_to']);
            }
            if (!empty($filters['class_id'])) {
                $this->db->where('s.class_id', $filters['class_id']);
            }
            if (!empty($filters['payment_status'])) {
                $this->db->where('ss.payment_status', $filters['payment_status']);
            }
            if (!empty($filters['student_search'])) {
                $this->db->group_start();
                $this->db->like('CONCAT(s.firstname, " ", s.lastname)', $filters['student_search']);
                $this->db->or_like('s.admission_no', $filters['student_search']);
                $this->db->group_end();
            }
            if (!empty($filters['sale_id'])) {
                $this->db->like('ss.id', $filters['sale_id']);
            }
            if (!empty($filters['item_search'])) {
                $this->db->join('inventory_items ii', 'ii.id = ssi.item_id', 'left');
                $this->db->like('ii.name', $filters['item_search']);
            }
            
            $this->db->group_by('ss.id');
            $this->db->order_by('ss.sale_date', 'DESC');
            
            if ($limit > 0) {
                $this->db->limit($limit, $offset);
            }
            
            $query = $this->db->get();
            
            if ($this->db->error()['code'] != 0) {
                error_log("Database error in getStudentSalesHistory: " . $this->db->error()['message']);
                return array();
            }
            
            return $query->result_array();
            
        } catch (Exception $e) {
            error_log("Exception in getStudentSalesHistory: " . $e->getMessage());
            return array();
        }
    }

    public function getStudentSalesCount($filters = array())
    {
        try {
            $this->db->select('COUNT(DISTINCT ss.id) as total');
            $this->db->from('student_sales ss');
            $this->db->join('students s', 's.id = ss.student_id', 'left');
            $this->db->join('student_sale_items ssi', 'ssi.sale_id = ss.id', 'left');
            
            // Apply same filters as history method
            if (!empty($filters['date_from'])) {
                $this->db->where('ss.sale_date >=', $filters['date_from']);
            }
            if (!empty($filters['date_to'])) {
                $this->db->where('ss.sale_date <=', $filters['date_to']);
            }
            if (!empty($filters['class_id'])) {
                $this->db->where('s.class_id', $filters['class_id']);
            }
            if (!empty($filters['payment_status'])) {
                $this->db->where('ss.payment_status', $filters['payment_status']);
            }
            if (!empty($filters['student_search'])) {
                $this->db->group_start();
                $this->db->like('CONCAT(s.firstname, " ", s.lastname)', $filters['student_search']);
                $this->db->or_like('s.admission_no', $filters['student_search']);
                $this->db->group_end();
            }
            if (!empty($filters['sale_id'])) {
                $this->db->like('ss.id', $filters['sale_id']);
            }
            if (!empty($filters['item_search'])) {
                $this->db->join('inventory_items ii', 'ii.id = ssi.item_id', 'left');
                $this->db->like('ii.name', $filters['item_search']);
            }
            
            $query = $this->db->get();
            
            if ($this->db->error()['code'] != 0) {
                error_log("Database error in getStudentSalesCount: " . $this->db->error()['message']);
                return 0;
            }
            
            $result = $query->row_array();
            return (int)$result['total'];
            
        } catch (Exception $e) {
            error_log("Exception in getStudentSalesCount: " . $e->getMessage());
            return 0;
        }
    }

    public function getStudentSalesSummary($filters = array())
    {
        try {
            $this->db->select('
                COUNT(DISTINCT ss.id) as total_sales,
                SUM(ss.total_amount) as total_revenue,
                SUM(ss.total_amount - ss.paid_amount) as pending_amount,
                COUNT(DISTINCT ss.student_id) as students_served
            ');
            $this->db->from('student_sales ss');
            $this->db->join('students s', 's.id = ss.student_id', 'left');
            $this->db->join('student_sale_items ssi', 'ssi.sale_id = ss.id', 'left');
            
            // Apply same filters as history method
            if (!empty($filters['date_from'])) {
                $this->db->where('ss.sale_date >=', $filters['date_from']);
            }
            if (!empty($filters['date_to'])) {
                $this->db->where('ss.sale_date <=', $filters['date_to']);
            }
            if (!empty($filters['class_id'])) {
                $this->db->where('s.class_id', $filters['class_id']);
            }
            if (!empty($filters['payment_status'])) {
                $this->db->where('ss.payment_status', $filters['payment_status']);
            }
            if (!empty($filters['student_search'])) {
                $this->db->group_start();
                $this->db->like('CONCAT(s.firstname, " ", s.lastname)', $filters['student_search']);
                $this->db->or_like('s.admission_no', $filters['student_search']);
                $this->db->group_end();
            }
            if (!empty($filters['sale_id'])) {
                $this->db->like('ss.id', $filters['sale_id']);
            }
            if (!empty($filters['item_search'])) {
                $this->db->join('inventory_items ii', 'ii.id = ssi.item_id', 'left');
                $this->db->like('ii.name', $filters['item_search']);
            }
            
            $query = $this->db->get();
            
            if ($this->db->error()['code'] != 0) {
                error_log("Database error in getStudentSalesSummary: " . $this->db->error()['message']);
                return array(
                    'total_sales' => 0,
                    'total_revenue' => 0,
                    'pending_amount' => 0,
                    'students_served' => 0
                );
            }
            
            $result = $query->row_array();
            return array(
                'total_sales' => (int)($result['total_sales'] ?? 0),
                'total_revenue' => (float)($result['total_revenue'] ?? 0),
                'pending_amount' => (float)($result['pending_amount'] ?? 0),
                'students_served' => (int)($result['students_served'] ?? 0)
            );
            
        } catch (Exception $e) {
            error_log("Exception in getStudentSalesSummary: " . $e->getMessage());
            return array(
                'total_sales' => 0,
                'total_revenue' => 0,
                'pending_amount' => 0,
                'students_served' => 0
            );
        }
    }

    public function getStudentSaleDetails($sale_id)
    {
        try {
            $this->db->select('
                ss.*,
                CONCAT(s.firstname, " ", s.lastname) as student_name,
                s.admission_no,
                c.class as class,
                sec.section as section
            ');
            $this->db->from('student_sales ss');
            $this->db->join('students s', 's.id = ss.student_id', 'left');
            $this->db->join('classes c', 'c.id = s.class_id', 'left');
            $this->db->join('sections sec', 'sec.id = s.section_id', 'left');
            $this->db->where('ss.id', $sale_id);
            
            $sale = $this->db->get()->row_array();
            
            if ($sale) {
                // Get sale items
                $this->db->select('
                    ssi.*,
                    ii.name as item_name,
                    ii.item_code,
                    ic.name as category_name
                ');
                $this->db->from('student_sale_items ssi');
                $this->db->join('inventory_items ii', 'ii.id = ssi.item_id', 'left');
                $this->db->join('inventory_categories ic', 'ic.id = ii.category_id', 'left');
                $this->db->where('ssi.sale_id', $sale_id);
                
                $sale['items'] = $this->db->get()->result_array();
            }
            
            return $sale;
            
        } catch (Exception $e) {
            error_log("Exception in getStudentSaleDetails: " . $e->getMessage());
            return false;
        }
    }
}
