/**
 * Camera Settings JavaScript
 * Handles camera configuration and settings integration for face attendance system
 */

class CameraSettings {
    constructor() {
        this.settings = null;
        this.currentStream = null;
        this.settingsLoaded = false;
        this.loadSettings();
    }
    
    /**
     * Load camera settings from server
     */
    async loadSettings() {
        try {
            // Get base URL from global variable or construct it
            let baseUrl;
            if (typeof base_url !== 'undefined') {
                baseUrl = base_url;
            } else {
                // Extract base URL from current location
                const pathParts = window.location.pathname.split('/');
                const adminIndex = pathParts.indexOf('admin');
                if (adminIndex > 0) {
                    baseUrl = window.location.origin + pathParts.slice(0, adminIndex).join('/') + '/';
                } else {
                    baseUrl = window.location.origin + '/';
                }
            }
            
            console.log('Constructed API URL:', baseUrl + 'admin/faceattendance/settings/getSettingsJson');
            const response = await fetch(baseUrl + 'admin/faceattendance/settings/getSettingsJson');
            
            // Check if response is ok
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            // Get response text first to debug
            const responseText = await response.text();
            console.log('Raw API response:', responseText);
            
            // Try to parse JSON
            const data = JSON.parse(responseText);
            
            if (data.status && data.settings) {
                this.settings = data.settings;
                this.settingsLoaded = true;
                console.log('Camera settings loaded:', this.settings);
            } else {
                console.error('Failed to load camera settings - invalid response format');
                this.setDefaultSettings();
            }
        } catch (error) {
            console.error('Error loading camera settings:', error);
            this.setDefaultSettings();
        }
    }
    
    /**
     * Wait for settings to be loaded
     */
    async waitForSettings() {
        return new Promise(resolve => {
            const checkSettings = () => {
                if (this.settingsLoaded || this.settings) {
                    resolve();
                } else {
                    setTimeout(checkSettings, 50);
                }
            };
            checkSettings();
        });
    }
    
    /**
     * Set default settings if loading fails
     */
    setDefaultSettings() {
        this.settings = {
            camera_preference: 'user_choice',
            face_match_threshold: 0.6,
            auto_capture_delay: 3,
            checkin_start_time: '08:00',
            checkin_end_time: '10:00',
            checkout_start_time: '15:00',
            checkout_end_time: '18:00',
            sms_notifications: 1,
            camera_resolution: 'auto'
        };
    }
    
    /**
     * Get camera constraints based on settings
     */
    getCameraConstraints() {
        if (!this.settings) {
            return { video: true };
        }
        
        let constraints = { video: {} };
        
        // Set camera preference
        switch (this.settings.camera_preference) {
            case 'front':
                constraints.video.facingMode = 'user';
                break;
            case 'back':
                // Try environment first, fallback to user if not available
                constraints.video.facingMode = { ideal: 'environment' };
                break;
            case 'environment':
                constraints.video.facingMode = { exact: 'environment' };
                break;
            case 'user_choice':
            default:
                // Let user choose or use default
                constraints.video = true;
                break;
        }
        
        // Set resolution
        switch (this.settings.camera_resolution) {
            case '480p':
                constraints.video.width = { ideal: 640 };
                constraints.video.height = { ideal: 480 };
                break;
            case '720p':
                constraints.video.width = { ideal: 1280 };
                constraints.video.height = { ideal: 720 };
                break;
            case '1080p':
                constraints.video.width = { ideal: 1920 };
                constraints.video.height = { ideal: 1080 };
                break;
            case 'auto':
            default:
                // Mobile optimized constraints - lower resolution for better performance
                const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
                
                constraints.video = {
                    width: isMobile ? { ideal: 480 } : 
                           this.settings.camera_resolution === 'auto' ? { ideal: 640 } : 
                           this.settings.camera_resolution === '720p' ? { ideal: 1280 } :
                           this.settings.camera_resolution === '480p' ? { ideal: 640 } : { ideal: 640 },
                    height: isMobile ? { ideal: 360 } :
                            this.settings.camera_resolution === 'auto' ? { ideal: 480 } : 
                            this.settings.camera_resolution === '720p' ? { ideal: 720 } :
                            this.settings.camera_resolution === '480p' ? { ideal: 480 } : { ideal: 480 },
                    frameRate: isMobile ? { ideal: 15, max: 20 } : { ideal: 30 }
                };
                // Use browser default
                break;
        }
        
        return constraints;
    }
    
    /**
     * Start camera with current settings
     */
    async startCamera(videoElement) {
        try {
            // Stop existing stream if any
            this.stopCamera();
            
            const constraints = this.getCameraConstraints();
            console.log('Starting camera with constraints:', constraints);
            
            // Check if user choice is selected and multiple cameras available
            if (this.settings.camera_preference === 'user_choice') {
                const devices = await this.getAvailableCameras();
                if (devices.length > 1) {
                    return await this.showCameraSelectionModal(videoElement, devices);
                }
            }
            
            const stream = await navigator.mediaDevices.getUserMedia(constraints);
            this.currentStream = stream;
            videoElement.srcObject = stream;
            
            return { success: true, stream: stream };
        } catch (error) {
            console.error('Error starting camera:', error);
            return { success: false, error: error.message };
        }
    }
    
    /**
     * Check if current time is within checkout time window
     */
    isWithinCheckoutTime() {
        if (!this.settings) {
            return true; // Allow if settings not loaded
        }
        
        const now = new Date();
        const currentTime = now.getHours().toString().padStart(2, '0') + ':' + now.getMinutes().toString().padStart(2, '0');
        
        const checkoutStart = this.settings.checkout_start_time || '15:00';
        const checkoutEnd = this.settings.checkout_end_time || '18:00';
        
        console.log('Time validation - Current:', currentTime, 'Checkout window:', checkoutStart, '-', checkoutEnd);
        
        return currentTime >= checkoutStart && currentTime <= checkoutEnd;
    }

    /**
     * Stop current camera stream
     */
    stopCamera() {
        if (this.currentStream) {
            this.currentStream.getTracks().forEach(track => track.stop());
            this.currentStream = null;
        }
    }
    
    /**
     * Get available cameras
     */
    async getAvailableCameras() {
        try {
            const devices = await navigator.mediaDevices.enumerateDevices();
            return devices.filter(device => device.kind === 'videoinput');
        } catch (error) {
            console.error('Error getting cameras:', error);
            return [];
        }
    }
    
    /**
     * Show camera selection modal for user choice
     */
    async showCameraSelectionModal(videoElement, cameras) {
        return new Promise((resolve) => {
            // Create modal HTML
            const modalHtml = `
                <div id="camera-selection-modal" style="
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(0,0,0,0.8);
                    z-index: 9999;
                    display: flex;
                    justify-content: center;
                    align-items: center;
                ">
                    <div style="
                        background: white;
                        border-radius: 15px;
                        padding: 30px;
                        max-width: 400px;
                        width: 90%;
                        text-align: center;
                        box-shadow: 0 20px 40px rgba(0,0,0,0.3);
                    ">
                        <h3 style="color: #667eea; margin-bottom: 20px;">
                            <i class="fa fa-camera" style="margin-right: 10px;"></i>
                            Select Camera
                        </h3>
                        <p style="color: #666; margin-bottom: 20px;">Choose which camera to use for face recognition:</p>
                        <div id="camera-list" style="margin-bottom: 20px;">
                            ${cameras.map((camera, index) => `
                                <button class="camera-option" data-device-id="${camera.deviceId}" style="
                                    display: block;
                                    width: 100%;
                                    padding: 12px;
                                    margin-bottom: 10px;
                                    border: 2px solid #e1e5e9;
                                    border-radius: 10px;
                                    background: #f8f9fa;
                                    cursor: pointer;
                                    transition: all 0.3s ease;
                                    font-size: 14px;
                                ">
                                    <i class="fa fa-video-camera" style="margin-right: 8px;"></i>
                                    ${camera.label || `Camera ${index + 1}`}
                                </button>
                            `).join('')}
                        </div>
                        <button id="cancel-camera-selection" style="
                            background: #dc3545;
                            color: white;
                            border: none;
                            padding: 10px 20px;
                            border-radius: 8px;
                            cursor: pointer;
                        ">Cancel</button>
                    </div>
                </div>
            `;
            
            // Add modal to page
            document.body.insertAdjacentHTML('beforeend', modalHtml);
            
            // Add event listeners
            document.querySelectorAll('.camera-option').forEach(button => {
                button.addEventListener('click', async () => {
                    const deviceId = button.dataset.deviceId;
                    document.getElementById('camera-selection-modal').remove();
                    
                    try {
                        const constraints = {
                            video: { deviceId: { exact: deviceId } }
                        };
                        
                        const stream = await navigator.mediaDevices.getUserMedia(constraints);
                        this.currentStream = stream;
                        videoElement.srcObject = stream;
                        
                        resolve({ success: true, stream: stream });
                    } catch (error) {
                        resolve({ success: false, error: error.message });
                    }
                });
                
                // Hover effects
                button.addEventListener('mouseenter', () => {
                    button.style.borderColor = '#667eea';
                    button.style.background = '#e3f2fd';
                });
                
                button.addEventListener('mouseleave', () => {
                    button.style.borderColor = '#e1e5e9';
                    button.style.background = '#f8f9fa';
                });
            });
            
            document.getElementById('cancel-camera-selection').addEventListener('click', () => {
                document.getElementById('camera-selection-modal').remove();
                resolve({ success: false, error: 'User cancelled camera selection' });
            });
        });
    }
    
    /**
     * Check if current time is within check-in window
     */
    async isCheckinAllowed() {
        try {
            const baseUrl = (typeof base_url !== 'undefined') ? base_url : 
                           (window.location.origin + window.location.pathname.split('/').slice(0, -2).join('/') + '/');
            const response = await fetch(baseUrl + 'admin/faceattendance/settings/isCheckinAllowed');
            const data = await response.json();
            return data;
        } catch (error) {
            console.error('Error checking check-in time:', error);
            return { status: false, allowed: true }; // Allow by default on error
        }
    }
    
    /**
     * Check if current time is within check-out window
     */
    async isCheckoutAllowed() {
        try {
            const baseUrl = (typeof base_url !== 'undefined') ? base_url : 
                           (window.location.origin + window.location.pathname.split('/').slice(0, -2).join('/') + '/');
            const response = await fetch(baseUrl + 'admin/faceattendance/settings/isCheckoutAllowed');
            const data = await response.json();
            return data;
        } catch (error) {
            console.error('Error checking check-out time:', error);
            return { status: false, allowed: true }; // Allow by default on error
        }
    }
    
    /**
     * Get face match threshold
     */
    getFaceMatchThreshold() {
        return this.settings ? this.settings.face_match_threshold : 0.6;
    }
    
    /**
     * Get auto capture delay
     */
    getAutoCaptureDelay() {
        return this.settings ? this.settings.auto_capture_delay * 1000 : 3000; // Convert to milliseconds
    }
    
    /**
     * Check if SMS notifications are enabled
     */
    isSmsEnabled() {
        return this.settings ? this.settings.sms_notifications == 1 : true;
    }
    
    /**
     * Validate time window before allowing attendance
     */
    async validateTimeWindow(isCheckout = false) {
        const timeCheck = isCheckout ? await this.isCheckoutAllowed() : await this.isCheckinAllowed();
        
        if (!timeCheck.allowed) {
            const windowType = isCheckout ? 'check-out' : 'check-in';
            const windowTime = isCheckout ? timeCheck.checkout_window : timeCheck.checkin_window;
            
            return {
                allowed: false,
                message: `${windowType} is only allowed during ${windowTime}. Current time: ${timeCheck.current_time}`
            };
        }
        
        return { allowed: true };
    }
    
    /**
     * Show time validation error
     */
    showTimeValidationError(message) {
        // Create and show time validation modal
        const modalHtml = `
            <div id="time-validation-modal" style="
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.8);
                z-index: 9999;
                display: flex;
                justify-content: center;
                align-items: center;
            ">
                <div style="
                    background: white;
                    border-radius: 15px;
                    padding: 30px;
                    max-width: 500px;
                    width: 90%;
                    text-align: center;
                    box-shadow: 0 20px 40px rgba(0,0,0,0.3);
                ">
                    <div style="
                        background: linear-gradient(135deg, #ff6b6b 0%, #ee5a24 100%);
                        color: white;
                        padding: 15px;
                        border-radius: 10px;
                        margin-bottom: 20px;
                    ">
                        <i class="fa fa-clock-o" style="font-size: 24px; margin-bottom: 10px;"></i>
                        <h3 style="margin: 0;">Time Window Restriction</h3>
                    </div>
                    <p style="color: #666; margin-bottom: 20px; font-size: 16px;">${message}</p>
                    <button onclick="document.getElementById('time-validation-modal').remove()" style="
                        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                        color: white;
                        border: none;
                        padding: 12px 25px;
                        border-radius: 8px;
                        cursor: pointer;
                        font-weight: 600;
                    ">
                        <i class="fa fa-check" style="margin-right: 8px;"></i>
                        Understood
                    </button>
                </div>
            </div>
        `;
        
        document.body.insertAdjacentHTML('beforeend', modalHtml);
    }
}

// Global camera settings instance
window.cameraSettings = new CameraSettings();

// Utility functions for time validation
window.validateCheckinTime = async function() {
    return await window.cameraSettings.validateTimeWindow(false);
};

window.validateCheckoutTime = async function() {
    return await window.cameraSettings.validateTimeWindow(true);
};

// Enhanced camera start function with settings integration
window.startCameraWithSettings = async function(videoElement) {
    return await window.cameraSettings.startCamera(videoElement);
};

// Enhanced face detection with settings threshold
window.detectFaceWithSettings = async function(video, faceapi) {
    const threshold = window.cameraSettings.getFaceMatchThreshold();
    
    return await faceapi.detectAllFaces(video, new faceapi.TinyFaceDetectorOptions({
        inputSize: 416,
        scoreThreshold: threshold
    })).withFaceLandmarks().withFaceDescriptors();
};

console.log('Camera Settings JavaScript loaded successfully');
