/**
 * Script to download face-api.js models
 * 
 * This script downloads the required face-api.js models from the GitHub repository
 * and saves them to the models directory.
 */

const fs = require('fs');
const path = require('path');
const https = require('https');

// Create models directory if it doesn't exist
const modelsDir = path.join(__dirname, 'models');
if (!fs.existsSync(modelsDir)) {
    fs.mkdirSync(modelsDir, { recursive: true });
}

// Models to download
const models = [
    'tiny_face_detector_model-weights_manifest.json',
    'tiny_face_detector_model-shard1',
    'face_landmark_68_model-weights_manifest.json',
    'face_landmark_68_model-shard1',
    'face_recognition_model-weights_manifest.json',
    'face_recognition_model-shard1',
    'face_expression_model-weights_manifest.json',
    'face_expression_model-shard1'
];

// Base URL for models
const baseUrl = 'https://raw.githubusercontent.com/justadudewhohacks/face-api.js/master/weights/';

// Download function
function downloadFile(url, dest) {
    return new Promise((resolve, reject) => {
        const file = fs.createWriteStream(dest);
        https.get(url, (response) => {
            response.pipe(file);
            file.on('finish', () => {
                file.close();
                console.log(`Downloaded: ${dest}`);
                resolve();
            });
        }).on('error', (err) => {
            fs.unlink(dest, () => {}); // Delete the file on error
            reject(err);
        });
    });
}

// Download all models
async function downloadModels() {
    console.log('Downloading face-api.js models...');
    
    for (const model of models) {
        const url = baseUrl + model;
        const dest = path.join(modelsDir, model);
        
        try {
            await downloadFile(url, dest);
        } catch (error) {
            console.error(`Error downloading ${model}:`, error);
        }
    }
    
    console.log('All models downloaded successfully!');
}

// Start download
downloadModels();
