// Simple face detection script
let video = null;
let canvas = null;
let ctx = null;
let isModelLoaded = false;
let detectionInterval = null;

// Initialize when document is ready
$(document).ready(function() {
    console.log('Simple face detection initializing...');
    
    // Add button to page
    if ($('#simple-face-btn').length === 0) {
        $('.box-header').append(`
            <button id="simple-face-btn" class="btn btn-warning pull-right" style="margin-right: 10px;">
                Try Simple Face Detection
            </button>
        `);
    }
    
    // Add container for video
    if ($('#simple-face-container').length === 0) {
        $('.qrcodediv').after(`
            <div id="simple-face-container" style="display:none; margin-top: 20px;">
                <div class="row">
                    <div class="col-md-6 col-md-offset-3">
                        <div class="panel panel-default">
                            <div class="panel-heading">
                                <h3 class="panel-title">Simple Face Detection</h3>
                            </div>
                            <div class="panel-body" style="text-align: center;">
                                <video id="simple-video" width="400" height="300" autoplay playsinline muted style="border: 1px solid #ddd;"></video>
                                <canvas id="simple-canvas" width="400" height="300" style="display:none; position: absolute; top: 0; left: 0;"></canvas>
                                <div id="simple-status" class="alert alert-info" style="margin-top: 10px;">Ready to start face detection</div>
                                <button id="simple-start-btn" class="btn btn-primary">Start Camera</button>
                                <button id="simple-stop-btn" class="btn btn-danger" style="display:none;">Stop Camera</button>
                                <button id="simple-mark-btn" class="btn btn-success" style="display:none;">Mark Attendance</button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `);
    }
    
    // Add event listeners
    $('#simple-face-btn').on('click', function() {
        $('#simple-face-container').toggle();
    });
    
    $('#simple-start-btn').on('click', function() {
        startCamera();
        $(this).hide();
        $('#simple-stop-btn, #simple-mark-btn').show();
    });
    
    $('#simple-stop-btn').on('click', function() {
        stopCamera();
        $(this).hide();
        $('#simple-mark-btn').hide();
        $('#simple-start-btn').show();
    });
    
    $('#simple-mark-btn').on('click', function() {
        markTestAttendance();
    });
    
    // Load face-api.js models
    loadModels();
});

// Load face-api.js models
async function loadModels() {
    try {
        updateStatus('Loading face detection models...');
        
        const modelPath = base_url + 'backend/faceattendance/models';
        console.log('Loading models from:', modelPath);
        
        // Load only the face detection model
        await faceapi.nets.tinyFaceDetector.loadFromUri(modelPath);
        
        console.log('Face detection model loaded successfully');
        isModelLoaded = true;
        updateStatus('Face detection model loaded. Click "Start Camera" to begin.');
    } catch (error) {
        console.error('Error loading models:', error);
        updateStatus('Error loading face detection models: ' + error.message);
    }
}

// Start camera
async function startCamera() {
    if (!isModelLoaded) {
        updateStatus('Face detection model not loaded yet. Please wait...');
        return;
    }
    
    updateStatus('Starting camera...');
    
    try {
        // Get video element
        video = document.getElementById('simple-video');
        canvas = document.getElementById('simple-canvas');
        ctx = canvas.getContext('2d');
        
        // Show canvas
        canvas.style.display = 'block';
        
        // Request camera access with simple constraints
        const stream = await navigator.mediaDevices.getUserMedia({ 
            video: true,
            audio: false
        });
        
        // Set video source
        video.srcObject = stream;
        
        // Wait for video to be ready
        video.onloadedmetadata = function() {
            video.play();
            updateStatus('Camera started. Looking for faces...');
            
            // Start face detection
            startDetection();
        };
    } catch (error) {
        console.error('Error starting camera:', error);
        updateStatus('Error accessing camera: ' + error.message);
    }
}

// Stop camera
function stopCamera() {
    // Stop detection interval
    if (detectionInterval) {
        clearInterval(detectionInterval);
        detectionInterval = null;
    }
    
    // Stop video stream
    if (video && video.srcObject) {
        const tracks = video.srcObject.getTracks();
        tracks.forEach(track => track.stop());
        video.srcObject = null;
    }
    
    // Clear canvas
    if (ctx) {
        ctx.clearRect(0, 0, canvas.width, canvas.height);
    }
    
    // Hide canvas
    if (canvas) {
        canvas.style.display = 'none';
    }
    
    updateStatus('Camera stopped.');
}

// Start face detection
function startDetection() {
    // Clear any existing interval
    if (detectionInterval) {
        clearInterval(detectionInterval);
    }
    
    // Start detection interval
    detectionInterval = setInterval(async () => {
        if (video && video.readyState === 4) {
            try {
                // Detect faces
                const detections = await faceapi.detectAllFaces(
                    video, 
                    new faceapi.TinyFaceDetectorOptions({
                        inputSize: 160,
                        scoreThreshold: 0.3
                    })
                );
                
                // Clear canvas
                ctx.clearRect(0, 0, canvas.width, canvas.height);
                
                // Draw detections
                if (detections.length > 0) {
                    console.log('Detected', detections.length, 'faces');
                    updateStatus('Detected ' + detections.length + ' faces');
                    
                    // Draw rectangles around faces
                    detections.forEach(detection => {
                        const box = detection.box;
                        ctx.strokeStyle = '#00ff00';
                        ctx.lineWidth = 3;
                        ctx.strokeRect(box.x, box.y, box.width, box.height);
                        
                        // Draw label
                        ctx.fillStyle = '#00ff00';
                        ctx.font = '16px Arial';
                        ctx.fillText('Face Detected', box.x, box.y - 5);
                    });
                } else {
                    updateStatus('No faces detected. Please position your face in the camera.');
                }
            } catch (error) {
                console.error('Error during face detection:', error);
                updateStatus('Error during face detection: ' + error.message);
            }
        }
    }, 200);
}

// Mark test attendance
function markTestAttendance() {
    // Call the attendance API with test data
    $.ajax({
        url: base_url + 'admin/faceattendance/attendance/markAttendance',
        type: 'POST',
        data: {
            id: '1', // Use first student ID
            type: 'student'
        },
        dataType: 'json',
        success: function(response) {
            if (response.status) {
                updateStatus('Test attendance marked successfully!');
                
                // Display profile data
                $('#profile_data').html(response.page);
                
                // Stop camera
                stopCamera();
                
                // Show reset button
                $('.scanner').removeClass('displayblock').addClass('displaynone');
                $('#qrbutton').removeClass('displaynone').addClass('displayblock');
            } else {
                updateStatus('Error marking test attendance: ' + response.msg);
            }
        },
        error: function(xhr, status, error) {
            updateStatus('Error marking test attendance: ' + error);
        }
    });
}

// Update status message
function updateStatus(message) {
    $('#simple-status').text(message);
    console.log('Status:', message);
}
