<!DOCTYPE html>
<?php
// Enable error reporting in the view
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Debug variables passed to the view
error_log('Variables in idcardpreview view: ' . json_encode(get_defined_vars()));

// Validate required variables
if (!isset($idcard) || !isset($student)) {
    error_log('Missing required variables in idcardpreview view');
    echo '<div style="color: red; padding: 20px; background: #ffeeee; border: 1px solid #ff0000;">';
    echo '<h3>Error: Missing Required Data</h3>';
    echo '<p>Some required data is missing to generate the ID card.</p>';
    if (!isset($idcard)) echo '<p>ID Card template data is missing</p>';
    if (!isset($student)) echo '<p>Student data is missing</p>';
    echo '</div>';
    exit;
}

// Debug template size values
error_log('ID Card Template Size Values (Single Preview):');
error_log('card_width: ' . (isset($idcard->card_width) ? $idcard->card_width : 'not set'));
error_log('card_height: ' . (isset($idcard->card_height) ? $idcard->card_height : 'not set'));
error_log('header_height: ' . (isset($idcard->header_height) ? $idcard->header_height : 'not set'));
error_log('footer_height: ' . (isset($idcard->footer_height) ? $idcard->footer_height : 'not set'));
error_log('content_height: ' . (isset($idcard->content_height) ? $idcard->content_height : 'not set'));
error_log('content_width: ' . (isset($idcard->content_width) ? $idcard->content_width : 'not set'));
error_log('enable_image_height: ' . (isset($idcard->enable_image_height) ? $idcard->enable_image_height : 'not set'));
?>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $this->lang->line('student_id_card'); ?></title>
    <style type="text/css">
        body {
            font-family: 'Arial', sans-serif;
            margin: 0;
            padding: 0;
            background-color: #f4f4f4;
        }
        /* Debug info for troubleshooting */
        .debug-info {
            background: #ffffcc;
            padding: 10px;
            margin: 10px 0;
            border: 1px solid #ffcc00;
            font-family: monospace;
            font-size: 12px;
        }
        .id-card-container {
            width: 100%;
            max-width: 1200px;
            margin: 20px auto;
            padding: 20px;
            background-color: #fff;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
            border-radius: 5px;
        }
        .id-card-header {
            text-align: center;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #ddd;
        }
        .id-card-header h2 {
            margin: 0;
            color: #333;
        }
        .id-card {
            position: relative;
            width: <?php echo $idcard->card_width; ?>px !important;
            height: <?php echo $idcard->card_height; ?>px !important;
            margin: 0 auto;
            border: 1px solid #ddd;
            box-shadow: 0 0 5px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            background-color: #fff;
            /* Force size to be respected */
            min-width: <?php echo $idcard->card_width; ?>px !important;
            min-height: <?php echo $idcard->card_height; ?>px !important;
            max-width: <?php echo $idcard->card_width; ?>px !important;
            max-height: <?php echo $idcard->card_height; ?>px !important;
            box-sizing: border-box !important;
        }
        .id-card-background {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 1;
        }
        .id-card-header-section {
            height: <?php echo $idcard->header_height; ?>px !important;
            min-height: <?php echo $idcard->header_height; ?>px !important;
            max-height: <?php echo $idcard->header_height; ?>px !important;
            background-color: #f8f8f8;
            padding: 5px;
            border-bottom: 1px solid #ddd;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: relative;
            z-index: 1;
            box-sizing: border-box !important;
        }
        .id-card-content {
            height: <?php echo $idcard->content_height; ?>px !important;
            min-height: <?php echo $idcard->content_height; ?>px !important;
            max-height: <?php echo $idcard->content_height; ?>px !important;
            width: <?php echo $idcard->content_width; ?>px !important;
            min-width: <?php echo $idcard->content_width; ?>px !important;
            max-width: <?php echo $idcard->content_width; ?>px !important;
            margin: 0 auto;
            display: flex;
            align-items: center;
            position: relative;
            z-index: 1;
            box-sizing: border-box !important;
        }
        .id-card-footer {
            height: <?php echo $idcard->footer_height; ?>px !important;
            min-height: <?php echo $idcard->footer_height; ?>px !important;
            max-height: <?php echo $idcard->footer_height; ?>px !important;
            background-color: #f8f8f8;
            border-top: 1px solid #ddd;
            padding: 5px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: absolute;
            bottom: 0;
            width: 100%;
            z-index: 1;
            box-sizing: border-box !important;
        }
        .id-card-header-left {
            position: absolute;
            top: 10px;
            left: 10px;
            font-size: 14px;
        }
        .id-card-header-center {
            position: absolute;
            top: 10px;
            left: 50%;
            transform: translateX(-50%);
            font-size: 14px;
            text-align: center;
        }
        .id-card-header-right {
            position: absolute;
            top: 10px;
            right: 10px;
            font-size: 14px;
            text-align: right;
        }
        .id-card-footer-left {
            position: absolute;
            bottom: 10px;
            left: 10px;
            font-size: 12px;
        }
        .id-card-footer-center {
            position: absolute;
            bottom: 10px;
            left: 50%;
            transform: translateX(-50%);
            font-size: 12px;
            text-align: center;
        }
        .id-card-footer-right {
            position: absolute;
            bottom: 10px;
            right: 10px;
            font-size: 12px;
            text-align: right;
        }
        .student-photo-container {
            position: absolute;
            z-index: 3;
            <?php 
            // Set position based on the selected position
            $imageTop = !empty($idcard->image_top) ? $idcard->image_top . 'px' : '20px';
            $imageSide = !empty($idcard->image_side) ? $idcard->image_side . 'px' : '20px';
            $position = !empty($idcard->image_position) ? $idcard->image_position : 'top-left';
            
            switch($position) {
                case 'top-right':
                    echo 'top: ' . $imageTop . '; right: ' . $imageSide . '; left: auto;';
                    break;
                case 'bottom-left':
                    echo 'bottom: ' . $imageTop . '; left: ' . $imageSide . '; top: auto;';
                    break;
                case 'bottom-right':
                    echo 'bottom: ' . $imageTop . '; right: ' . $imageSide . '; left: auto; top: auto;';
                    break;
                default: // top-left
                    echo 'top: ' . $imageTop . '; left: ' . $imageSide . ';';
            }
            ?>
        }
        .student-photo {
            width: <?php echo (!empty($idcard->image_width) ? $idcard->image_width . 'px' : 'auto'); ?> !important;
            height: <?php echo (!empty($idcard->enable_image_height) ? $idcard->enable_image_height . 'px' : '120px'); ?> !important;
            min-height: <?php echo (!empty($idcard->enable_image_height) ? $idcard->enable_image_height . 'px' : '120px'); ?> !important;
            max-height: <?php echo (!empty($idcard->enable_image_height) ? $idcard->enable_image_height . 'px' : '120px'); ?> !important;
            border: 1px solid #ddd;
            background-color: #f5f5f5;
            padding: 3px;
            border-radius: 3px;
            box-sizing: border-box !important;
            object-fit: cover;
        }
        .qrcode-container {
            position: absolute;
            right: 20px;
            bottom: 20px;
            z-index: 3;
        }
        .qrcode {
            width: 60px;
            height: 60px;
            border: 1px solid #ddd;
            background-color: #fff;
            padding: 3px;
            border-radius: 3px;
            box-sizing: border-box !important;
        }
        .id-card-text {
            padding: 10px;
            text-align: left;
            font-size: 14px;
            line-height: 1.5;
        }
        .print-button {
            display: block;
            width: 200px;
            margin: 20px auto;
            padding: 10px;
            background-color: #3c8dbc;
            color: #fff;
            text-align: center;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
        }
        .print-button:hover {
            background-color: #367fa9;
        }
        @media print {
            body {
                background-color: #fff;
            }
            .id-card-container {
                box-shadow: none;
                padding: 0;
            }
            .print-button {
                display: none;
            }
            .id-card {
                page-break-inside: avoid;
            }
        }
    </style>
</head>
<body>
    <!-- Debug info for troubleshooting -->
    <div class="debug-info">
        <h4>ID Card Template Size Values:</h4>
        <ul>
            <li>Card Width: <?php echo $idcard->card_width; ?>px</li>
            <li>Card Height: <?php echo $idcard->card_height; ?>px</li>
            <li>Header Height: <?php echo $idcard->header_height; ?>px</li>
            <li>Footer Height: <?php echo $idcard->footer_height; ?>px</li>
            <li>Content Height: <?php echo $idcard->content_height; ?>px</li>
            <li>Content Width: <?php echo $idcard->content_width; ?>px</li>
            <li>Image Height: <?php echo $idcard->enable_image_height; ?>px</li>
        </ul>
    </div>
    <div class="id-card-container">
        <div class="id-card-header">
            <h2><?php echo $this->lang->line('student_id_card'); ?></h2>
            <button class="btn btn-primary print-button" style="margin-top: 10px;" onclick="window.print();">
                <i class="fa fa-print"></i> Print ID Card
            </button>
            <script>
                // Add event listener as a fallback
                document.querySelector('.print-button').addEventListener('click', function() {
                    window.print();
                });
            </script>
        </div>
        
        <div class="id-card">
            <?php 
            // Debug information
            error_log('Background image: ' . (isset($idcard->background_image) ? $idcard->background_image : 'not set'));
            
            // Use the background image path passed from the controller if available
            if(isset($bg_image_path) && !empty($bg_image_path)){ 
                error_log('Using background image path from controller: ' . $bg_image_path);
            ?>
            <img class="id-card-background" src="<?php echo $bg_image_path; ?>" alt="Background" onerror="this.style.display='none'; console.log('Failed to load background image: ' + this.src);">
            <?php } else {
                // Get background image based on whether $idcard is array or object
                $background_image = '';
                if (is_array($idcard)) {
                    $background_image = isset($idcard['background_image']) ? $idcard['background_image'] : '';
                } else {
                    $background_image = isset($idcard->background_image) ? $idcard->background_image : '';
                }
                
                if(!empty($background_image)) {
                    $bg_image_path = base_url('uploads/idcard/') . $background_image;
                    error_log('Using calculated background image path: ' . $bg_image_path);
            ?>
            <img class="id-card-background" src="<?php echo $bg_image_path; ?>" alt="Background" onerror="this.style.display='none'; console.log('Failed to load background image: ' + this.src);">
            <?php } else { ?>
            <!-- No background image specified -->
            <div style="position: absolute; top: 10px; left: 10px; color: #999; font-size: 10px;">No background image</div>
            <?php } } ?>
            
            <div class="id-card-header-section">
                <div class="id-card-header-left"><?php echo processPlaceholders(is_array($idcard) ? $idcard['left_header'] : $idcard->left_header, $student); ?></div>
                <div class="id-card-header-center"><?php echo processPlaceholders(is_array($idcard) ? $idcard['center_header'] : $idcard->center_header, $student); ?></div>
                <div class="id-card-header-right"><?php echo processPlaceholders(is_array($idcard) ? $idcard['right_header'] : $idcard->right_header, $student); ?></div>
            </div>
            
            <div class="id-card-content">
                <?php 
                // Debug information for student image
                $enable_student_image = is_array($idcard) ? (isset($idcard['enable_student_image']) ? $idcard['enable_student_image'] : 0) : (isset($idcard->enable_student_image) ? $idcard->enable_student_image : 0);
                error_log('Student image enabled: ' . $enable_student_image);
                
                if($enable_student_image == 1){ 
                    // Get the image URL
                    $student_image = '';
                    
                    // 1. Try to use the preloaded image_url from the student object
                    if(isset($student->image_url) && !empty($student->image_url)) {
                        $student_image = $student->image_url;
                        error_log('Using preloaded student image URL: ' . $student_image);
                    } 
                    // 2. Fallback to student_image_path if available
                    else if(isset($student_image_path) && !empty($student_image_path)) {
                        $student_image = $student_image_path;
                        error_log('Using student image path from controller: ' . $student_image);
                    } 
                    // 3. Final fallback to the getStudentImagePath function
                    else {
                        $student_image = getStudentImagePath($student);
                        error_log('Using calculated student image path: ' . $student_image);
                    }
                    
                    // Set default image if no image found
                    if(empty($student_image) || !filter_var($student_image, FILTER_VALIDATE_URL)) {
                        $student_image = base_url('uploads/student_images/no_image.png');
                    }
                    
                    // Ensure the URL is absolute
                    if (strpos($student_image, 'http') !== 0) {
                        if (strpos($student_image, '/') !== 0) {
                            $student_image = '/' . $student_image;
                        }
                        $student_image = rtrim(base_url(), '/') . $student_image;
                    }
                    
                    // Get image dimensions from template or use defaults
                    $image_width = isset($idcard->image_width) && !empty($idcard->image_width) ? $idcard->image_width : 100;
                    $image_height = isset($idcard->enable_image_height) && !empty($idcard->enable_image_height) ? $idcard->enable_image_height : 120;
                    $image_top = isset($idcard->image_top) && $idcard->image_top !== '' ? $idcard->image_top . 'px' : '20px';
                    $image_side = isset($idcard->image_left) && $idcard->image_left !== '' ? $idcard->image_left . 'px' : '20px';
                ?>
                <div class="student-photo-container" 
                     style="
                        position: absolute;
                        width: <?php echo $image_width; ?>px !important;
                        height: <?php echo $image_height; ?>px !important;
                        top: <?php echo $image_top; ?> !important;
                        left: <?php echo $image_side; ?> !important;
                        border: 1px solid #ddd; 
                        overflow: hidden; 
                        display: flex; 
                        justify-content: center; 
                        align-items: center; 
                        background-color: #f5f5f5;
                        z-index: 10;
                     ">
                    <img src="<?php echo htmlspecialchars($student_image); ?>" 
                         alt="Student Photo" 
                         style="
                            max-width: 100%; 
                            max-height: 100%; 
                            width: auto;
                            height: auto;
                            object-fit: contain;" 
                         onerror="this.onerror=null; this.src='<?php echo base_url('uploads/student_images/no_image.png'); ?>';">
                    <!-- Debug information (only in development) -->
                    <?php if(ENVIRONMENT === 'development'): ?>
                    <div style="
                        position: absolute;
                        bottom: 0;
                        left: 0;
                        right: 0;
                        background: rgba(0,0,0,0.5);
                        color: white;
                        font-size: 8px;
                        padding: 2px;
                        text-align: center;
                        white-space: nowrap;
                        overflow: hidden;
                        text-overflow: ellipsis;">
                        <?php echo htmlspecialchars(basename($student_image)); ?>
                    </div>
                    <?php endif; ?>
                </div>
                <?php } ?>
                
                <?php 
                // QR Code rendering logic
                $enable_qr_code = is_array($idcard) ? (isset($idcard['enable_qr_code']) ? $idcard['enable_qr_code'] : 0) : (isset($idcard->enable_qr_code) ? $idcard->enable_qr_code : 0);
                
                if($enable_qr_code == 1){ 
                    // Set default values if not set
                    $qrSize = is_array($idcard) ? 
                        (!empty($idcard['qr_size']) ? $idcard['qr_size'] . 'px' : '100px') : 
                        (!empty($idcard->qr_size) ? $idcard->qr_size . 'px' : '100px');
                    
                    $qrTop = is_array($idcard) ? 
                        (!empty($idcard['qr_top']) ? $idcard['qr_top'] . 'px' : '20px') : 
                        (!empty($idcard->qr_top) ? $idcard->qr_top . 'px' : '20px');
                    
                    $qrSide = is_array($idcard) ? 
                        (!empty($idcard['qr_side']) ? $idcard['qr_side'] . 'px' : '20px') : 
                        (!empty($idcard->qr_side) ? $idcard->qr_side . 'px' : '20px');
                    
                    // Set position based on selected position
                    $qrPositionStyle = '';
                    $qrPosition = is_array($idcard) ? 
                        (!empty($idcard['qr_position']) ? $idcard['qr_position'] : 'top-left') : 
                        (!empty($idcard->qr_position) ? $idcard->qr_position : 'top-left');
                    
                    switch($qrPosition) {
                        case 'top-right':
                            $qrPositionStyle = 'top: ' . $qrTop . '; right: ' . $qrSide . '; left: auto;';
                            break;
                        case 'bottom-left':
                            $qrPositionStyle = 'bottom: ' . $qrTop . '; left: ' . $qrSide . '; top: auto;';
                            break;
                        case 'bottom-right':
                            $qrPositionStyle = 'bottom: ' . $qrTop . '; right: ' . $qrSide . '; left: auto; top: auto;';
                            break;
                        default: // top-left
                            $qrPositionStyle = 'top: ' . $qrTop . '; left: ' . $qrSide . ';';
                    }
                    
                    // Get admission number from student data
                    $admissionNo = is_array($student) ? 
                        (isset($student['admission_no']) ? $student['admission_no'] : 'N/A') : 
                        (isset($student->admission_no) ? $student->admission_no : 'N/A');
                    
                    error_log('QR Code enabled for student: ' . $admissionNo . ', Position: ' . $qrPosition . ', Size: ' . $qrSize);
                ?>
                <div class="qr-code" style="
                    position: absolute;
                    <?php echo $qrPositionStyle; ?>
                    width: <?php echo $qrSize; ?>;
                    height: <?php echo $qrSize; ?>;
                    border: 1px solid #ddd;
                    background: #fff;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    z-index: 15;">
                    <img src="https://api.qrserver.com/v1/create-qr-code/?size=<?php echo str_replace('px', '', $qrSize); ?>x<?php echo str_replace('px', '', $qrSize); ?>&data=<?php echo urlencode($admissionNo); ?>" 
                         style="width: 100%; height: 100%; object-fit: contain;" 
                         alt="QR Code for <?php echo htmlspecialchars($admissionNo); ?>"
                         onerror="this.style.display='none'; console.log('Failed to load QR code');">
                </div>
                <?php } ?>
                
                <div class="id-card-text">
                    <?php echo processPlaceholders(is_array($idcard) ? $idcard['idcard_text'] : $idcard->idcard_text, $student); ?>
                </div>
            </div>
            
            <div class="id-card-footer">
                <div class="id-card-footer-left"><?php echo processPlaceholders(is_array($idcard) ? $idcard['left_footer'] : $idcard->left_footer, $student); ?></div>
                <div class="id-card-footer-center"><?php echo processPlaceholders(is_array($idcard) ? $idcard['center_footer'] : $idcard->center_footer, $student); ?></div>
                <div class="id-card-footer-right"><?php echo processPlaceholders(is_array($idcard) ? $idcard['right_footer'] : $idcard->right_footer, $student); ?></div>
            </div>
        </div>
        
        <button class="print-button" onclick="window.print()"><?php echo $this->lang->line('print'); ?></button>
    </div>
    
    <script type="text/javascript">
    // Apply template sizes directly to the ID card using JavaScript
    document.addEventListener('DOMContentLoaded', function() {
        // Get the ID card
        var idCard = document.querySelector('.id-card');
        
        // Template sizes from PHP
        var cardWidth = <?php echo $idcard->card_width; ?>;
        var cardHeight = <?php echo $idcard->card_height; ?>;
        var headerHeight = <?php echo $idcard->header_height; ?>;
        var contentHeight = <?php echo $idcard->content_height; ?>;
        var contentWidth = <?php echo $idcard->content_width; ?>;
        var footerHeight = <?php echo $idcard->footer_height; ?>;
        var imageHeight = <?php echo $idcard->enable_image_height; ?>;
        
        console.log('Template sizes:', {
            cardWidth: cardWidth,
            cardHeight: cardHeight,
            headerHeight: headerHeight,
            contentHeight: contentHeight,
            contentWidth: contentWidth,
            footerHeight: footerHeight,
            imageHeight: imageHeight
        });
        
        if (idCard) {
            // Force card size
            idCard.style.width = cardWidth + 'px';
            idCard.style.height = cardHeight + 'px';
            idCard.style.minWidth = cardWidth + 'px';
            idCard.style.minHeight = cardHeight + 'px';
            idCard.style.maxWidth = cardWidth + 'px';
            idCard.style.maxHeight = cardHeight + 'px';
            idCard.style.boxSizing = 'border-box';
            
            // Force header size
            var header = idCard.querySelector('.id-card-header-section');
            if (header) {
                header.style.height = headerHeight + 'px';
                header.style.minHeight = headerHeight + 'px';
                header.style.maxHeight = headerHeight + 'px';
                header.style.boxSizing = 'border-box';
            }
            
            // Force content size
            var content = idCard.querySelector('.id-card-content');
            if (content) {
                content.style.height = contentHeight + 'px';
                content.style.minHeight = contentHeight + 'px';
                content.style.maxHeight = contentHeight + 'px';
                content.style.width = contentWidth + 'px';
                content.style.minWidth = contentWidth + 'px';
                content.style.maxWidth = contentWidth + 'px';
                content.style.boxSizing = 'border-box';
            }
            
            // Force footer size
            var footer = idCard.querySelector('.id-card-footer');
            if (footer) {
                footer.style.height = footerHeight + 'px';
                footer.style.minHeight = footerHeight + 'px';
                footer.style.maxHeight = footerHeight + 'px';
                footer.style.boxSizing = 'border-box';
            }
            
            // Force image size
            var image = idCard.querySelector('.student-photo');
            if (image) {
                image.style.height = imageHeight + 'px';
                image.style.minHeight = imageHeight + 'px';
                image.style.maxHeight = imageHeight + 'px';
                image.style.boxSizing = 'border-box';
            }
        }
    });
    </script>
</body>
</html>

<?php
/**
 * Process placeholders in the ID card text
 * 
 * @param string $text The text containing placeholders
 * @param object|array $student Student data
 * @return string Processed text with placeholders replaced
 */
function processPlaceholders($text, $student) {
    if (empty($text)) {
        return '';
    }
    
    // Debug the input
    $debug = false;
    if ($debug) {
        error_log('Processing placeholders for text: ' . substr($text, 0, 500) . '...');
        error_log('Student data type: ' . gettype($student));
    }
    
    // Convert object to array if needed
    if (is_object($student)) {
        $student = (array) $student;
    } elseif (!is_array($student)) {
        error_log('Invalid student data provided to processPlaceholders');
        return $text;
    }
    
    if ($debug) {
        error_log('Student data keys: ' . implode(', ', array_keys($student)));
    }
    
    // Define all possible placeholders and their corresponding student data keys
    $placeholders = [
        // Basic student info
        '[name]' => ['firstname', 'lastname'],
        '[firstname]' => 'firstname',
        '[lastname]' => 'lastname',
        '[fullname]' => ['firstname', 'middlename', 'lastname'],
        
        // Academic info
        '[admission_no]' => 'admission_no',
        '[roll_no]' => 'roll_no',
        '[class]' => 'class',
        '[section]' => 'section',
        
        // Personal info
        '[dob]' => 'dob',
        '[gender]' => 'gender',
        '[blood_group]' => 'blood_group',
        
        // Contact info
        '[present_address]' => 'current_address',
        '[permanent_address]' => 'permanent_address',
        '[guardian]' => ['guardian_name', 'father_name', 'mother_name'],
        '[father_name]' => 'father_name',
        '[mother_name]' => 'mother_name',
        '[guardian_name]' => 'guardian_name',
        
        // Dates
        '[admission_date]' => 'admission_date',
        '[created_at]' => 'created_at',
        
        // Additional fields
        '[category]' => 'category',
        '[cast]' => 'cast',
        '[religion]' => 'religion',
        '[mobile_no]' => 'mobileno',
        '[email]' => 'email',
        
        // School info
        '[school_name]' => 'school_name',
        '[school_address]' => 'school_address',
        '[school_phone]' => 'school_phone',
        '[school_email]' => 'school_email',
        
        // Academic year
        '[academic_year]' => 'session_id', // This might need special handling
        
        // ID card specific
        '[id_card_number]' => 'id_card_no',
        '[valid_until]' => 'valid_until',
    ];
    
    // Process each placeholder
    foreach ($placeholders as $placeholder => $keys) {
        if (strpos($text, $placeholder) === false) {
            continue; // Skip if placeholder not found in text
        }
        
        $value = '';
        
        // Handle multiple possible keys (fallback chain)
        $keyList = is_array($keys) ? $keys : [$keys];
        
        foreach ($keyList as $key) {
            if (isset($student[$key]) && !empty($student[$key])) {
                $value = $student[$key];
                break;
            }
        }
        
        // Special handling for specific placeholders
        switch ($placeholder) {
            case '[fullname]':
                $parts = [];
                if (!empty($student['firstname'])) $parts[] = $student['firstname'];
                if (!empty($student['middlename'])) $parts[] = $student['middlename'];
                if (!empty($student['lastname'])) $parts[] = $student['lastname'];
                $value = implode(' ', $parts);
                break;
                
            case '[name]':
                $parts = [];
                if (!empty($student['firstname'])) $parts[] = $student['firstname'];
                if (!empty($student['lastname'])) $parts[] = $student['lastname'];
                $value = implode(' ', $parts);
                break;
                
            case '[guardian]':
                // Try to find any available guardian name
                if (empty($value)) {
                    if (!empty($student['father_name'])) {
                        $value = 'Father: ' . $student['father_name'];
                    } elseif (!empty($student['mother_name'])) {
                        $value = 'Mother: ' . $student['mother_name'];
                    } elseif (!empty($student['guardian_name'])) {
                        $value = 'Guardian: ' . $student['guardian_name'];
                    }
                }
                break;
                
            case '[school_name]':
                if (empty($value) && defined('SCHOOL_NAME')) {
                    $value = SCHOOL_NAME;
                }
                break;
                
            case '[academic_year]':
                // This might need to be replaced with actual academic year logic
                if (empty($value)) {
                    $value = date('Y') . '-' . (date('Y') + 1);
                }
                break;
        }
        
        // If value is still empty, show a placeholder for debugging
        if (empty($value) && $debug) {
            $value = '[' . str_replace(['[', ']'], '', $placeholder) . '_NOT_FOUND]';
        }
        
        // Replace the placeholder with the actual value
        $text = str_replace($placeholder, $value, $text);
    }
    
    // Clean up any remaining placeholders
    $text = preg_replace('/\[.*?\]/', '', $text);
    
    // Trim any extra whitespace
    $text = preg_replace('/\s+/', ' ', $text);
    $text = trim($text);
    
    if ($debug) {
        error_log('Processed text: ' . $text);
    }
    
    return $text;
}

/**
 * Get student image path with fallbacks
 * 
 * @param array|object $student Student data
 * @return string Image URL
 */
function getStudentImagePath($student) {
    // Enable error reporting for debugging
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    
    // Get CI instance
    $CI =& get_instance();
    $CI->load->helper('file');
    $CI->load->library('media_storage');
    
    // Start debug log
    error_log('\n' . str_repeat('=', 50));
    error_log('DEBUG: getStudentImagePath() called');
    error_log('Time: ' . date('Y-m-d H:i:s'));
    error_log('Memory Usage: ' . memory_get_usage() / 1024 / 1024 . ' MB');
    
    // Convert object to array if needed
    $is_object = is_object($student);
    if ($is_object) {
        $student = get_object_vars($student);
    }
    
    // Define possible directories to search for images
    $possible_dirs = [
        'uploads/student_images/',
        'uploads/student_images/student/',
        'uploads/images/student_images/',
        'uploads/student/profile/',
        'uploads/students/',
        'uploads/student/',
        'uploads/student_pics/',
        'uploads/student_photos/'
    ];
    
    // First, check if we have a direct image reference in the student data
    $image_fields = ['image', 'student_image', 'photo', 'picture', 'image_thumb', 'photo_thumb'];
    
    foreach ($image_fields as $field) {
        if (!empty($student[$field])) {
            $image = $student[$field];
            
            // If it's a full URL, return it directly
            if (filter_var($image, FILTER_VALIDATE_URL)) {
                return $image;
            }
            
            // Check in various directories
            foreach ($possible_dirs as $dir) {
                $file_path = FCPATH . $dir . $image;
                if (file_exists($file_path)) {
                    error_log('Found image at: ' . $file_path);
                    return base_url($dir . $image);
                }
            }
            
            // If we have a value but file doesn't exist, try to find by admission number
            break;
        }
    }
    
    // If still no image, try to find by admission number or student ID
    $identifiers = [];
    if (!empty($student['admission_no'])) $identifiers[] = $student['admission_no'];
    if (!empty($student['id'])) $identifiers[] = $student['id'];
    
    if (!empty($identifiers)) {
        error_log('Trying to find image by identifiers: ' . implode(', ', $identifiers));
        
        $possible_extensions = ['jpg', 'jpeg', 'png', 'gif'];
        
        foreach ($identifiers as $id) {
            foreach ($possible_dirs as $directory) {
                // Check for exact filename match
                foreach ($possible_extensions as $ext) {
                    $filename = $id . '.' . $ext;
                    $file_path = FCPATH . $directory . $filename;
                    
                    if (file_exists($file_path)) {
                        error_log('Found image by ID: ' . $file_path);
                        return base_url($directory . $filename);
                    }
                }
                
                // Check for files that start with the ID
                if (is_dir(FCPATH . $directory)) {
                    $files = scandir(FCPATH . $directory);
                    foreach ($files as $file) {
                        if ($file != '.' && $file != '..' && strpos($file, (string)$id) === 0) {
                            $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                            if (in_array($ext, $possible_extensions)) {
                                $file_path = FCPATH . $directory . $file;
                                if (file_exists($file_path)) {
                                    error_log('Found image with ID in filename: ' . $file_path);
                                    return base_url($directory . $file);
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    
    // Log all the directories we checked for debugging
    error_log('Failed to find student image. Checked the following directories for images:');
    foreach ($possible_dirs as $dir) {
        $full_path = FCPATH . $dir;
        $exists = is_dir($full_path) ? 'exists' : 'does not exist';
        error_log("- $full_path ($exists)");
        
        if (is_dir($full_path)) {
            $files = @scandir($full_path);
            if ($files === false) {
                error_log("  Could not read directory");
            } else {
                $file_count = count($files) - 2; // Subtract . and ..
                error_log("  Contains $file_count files");
                
                // Log first few files for debugging
                if ($file_count > 0) {
                    $files = array_diff($files, ['.', '..']);
                    $files = array_slice($files, 0, 5);
                    error_log("  Sample files: " . implode(', ', $files) . ($file_count > 5 ? '...' : ''));
                }
            }
        }
    }
    
    // Try to use default image from the first existing directory
    $default_image = null;
    foreach ($possible_dirs as $dir) {
        $default_path = FCPATH . $dir . 'no_image.png';
        if (file_exists($default_path)) {
            $default_image = base_url($dir . 'no_image.png');
            error_log('Found default image at: ' . $default_path);
            break;
        }
    }
    
    // If no default image found in any directory, use a simple SVG placeholder
    if ($default_image === null) {
        error_log('No default image found in any directory, using SVG placeholder');
        $default_image = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTUwIiBoZWlnaHQ9IjIwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiBzdHlsZT0iYmFja2dyb3VuZC1jb2xvcjojZWVlO2NvbG9yOiM5OTk7dGV4dC1hbGlnbjpjZW50ZXI7ZGlzcGxheTpmbGV4O2FsaWduLWl0ZW1zOmNlbnRlcjtqdXN0aWZ5LWNvbnRlbnQ6Y2VudGVyOyI+PHRleHQgeD0iNTAiIHk9IjEwMCIgZG9taW5hbnQtYmFzZWxpbmU9Im1pZGRsZSIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjE0IiBmb250LXdlaWdodD0iYm9sZCI+Tm8gUGhvdG8gQXZhaWxhYmxlPC90ZXh0Pjwvc3ZnPg=';
    }
    
    error_log('Using image: ' . $default_image);
    return $default_image;
}
?>
